<?php

/**
 * This trait provides helper methods related to EDD conversions tracking.
 */
trait Analytify_EDD_Conversions_Helpers {

	/**
	 * @var int|null The EDD order ID associated with the conversion.
	 */
	public $edd_order_id;

	/**
	 * @var EDD_Order|null The EDD order object associated with the conversion.
	 */
	public $edd_order;

	/**
	 * Checks if the current page is the order received page.
	 *
	 * @return bool Returns true if the current page is the order received page, false otherwise.
	 */
	public function conversion_checks() {

		if ( function_exists( 'edd_is_success_page' ) && edd_is_success_page() ) {
			return true;
		}
		return false;
	}

	/**
	 * Checks if the conversion is tracked for the current order.
	 *
	 * @return bool Returns true if the conversion is tracked, false otherwise.
	 */
	public function is_tracked() {
		if ( $this->get_order() ) {
			$tracked = $this->get_order()->get_meta( 'analytify_ads_conversion_tracked' );
		}
		return ! empty( $tracked ) && 'yes' === $tracked;
	}

	/**
	 * Retrieves the EDD order object associated with the current order number.
	 *
	 * @return bool|EDD_Payment Returns the EDD order object if found, otherwise false.
	 */
	public function get_order() {

		if ( empty( $this->edd_order ) ) {

			if ( $this->get_order_number() ) {

				$this->edd_order = edd_get_payment( $this->edd_order_id );

			} else {

				$this->edd_order = false;

			}

		}

		return $this->edd_order;

	}

	/**
	 * Retrieves the EDD order number from the query variables.
	 *
	 * @return bool|int Returns the EDD order number if found, otherwise false.
	 */
	public function get_order_number() {

		if ( isset( $this->edd_order_id ) ) {
			return $this->edd_order_id;
		}

		$edd_session = edd_get_purchase_session();

		if ( empty( $edd_session['purchase_key'] ) ) {
			return false;
		}

		$payment_key = $edd_session['purchase_key'];

		$edd_order_id    = edd_get_purchase_id_by_key( $payment_key );

		if ( ! empty( $edd_order_id ) && 0 !== $edd_order_id ) {
			$this->edd_order_id = $edd_order_id;
		}

		return $this->edd_order_id;

	}

	/**
	 * Retrieves the total amount of the current order.
	 *
	 * @return bool|string Returns the order total amount if the order is available, otherwise false.
	 */
	public function get_order_total() {

		if ( $this->get_order_number() ) {
			return edd_get_payment_amount( $this->edd_order_id );
		}

		return false;
	}

	/**
	 * Retrieves the currency of the current order.
	 *
	 * @return bool|string Returns the order currency if the order is available, otherwise false.
	 */
	public function get_order_currency() {

		if ( $this->get_order_number() ) {
			return edd_get_payment_currency_code( $this->edd_order_id );
		}

		return false;
	}

	/**
	 * Sets the conversion tracked status for the current order.
	 */
	public function set_order_tracked() {

		if ( $this->get_order_number() ) {
			$edd_order = $this->get_order();
			$edd_order->update_meta( 'analytify_ads_conversion_tracked', 'yes' );
		}

	}
}
