<?php
/**
 * Plugin Name: SityCloud AI Support Tickets (Network)
 * Description: نظام تذاكر دعم داخل شبكة ووردبريس مع رد تلقائي أولي من AI + لوحة شبكة لإدارة التذاكر + شورت كود للعميل.
 * Author: Sity Cloud
 * Version: 1.0
 * Network: true
 */

if (!defined('ABSPATH')) {
    exit;
}

class SityCloud_AI_Support_Tickets {

    private static $instance = null;
    private $tickets_table;
    private $messages_table;

    public static function instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        global $wpdb;
        // نستخدم base_prefix بحيث التذاكر تكون مشتركة بين الشبكة كلها
        $this->tickets_table   = $wpdb->base_prefix . 'sity_ai_tickets';
        $this->messages_table  = $wpdb->base_prefix . 'sity_ai_ticket_messages';

        register_activation_hook(__FILE__, array($this, 'activate'));
        add_action('network_admin_menu', array($this, 'register_network_menu'));
        add_shortcode('sity_support_tickets', array($this, 'shortcode_render'));

        // هندل فورمات الفرونت والادمن
        add_action('init', array($this, 'handle_front_post_requests'));
        add_action('admin_init', array($this, 'handle_admin_post_requests'));

        // تحميل ستايل بسيط في الادمن (لنفس صفحة البلجن فقط)
        add_action('admin_enqueue_scripts', array($this, 'admin_styles'));
    }

    /* -------------------- Activation: Create Tables -------------------- */

    public function activate() {
        global $wpdb;
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $charset_collate = $wpdb->get_charset_collate();

        $sql_tickets = "CREATE TABLE {$this->tickets_table} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            site_id BIGINT(20) UNSIGNED NOT NULL,
            user_id BIGINT(20) UNSIGNED NOT NULL,
            subject VARCHAR(255) NOT NULL,
            department VARCHAR(100) DEFAULT '',
            priority VARCHAR(50) DEFAULT 'normal',
            status VARCHAR(50) DEFAULT 'open',
            needs_human TINYINT(1) DEFAULT 0,
            created_at DATETIME NOT NULL,
            updated_at DATETIME NOT NULL,
            last_reply_by VARCHAR(20) DEFAULT 'client',
            PRIMARY KEY (id),
            KEY site_id (site_id),
            KEY user_id (user_id),
            KEY status (status)
        ) $charset_collate;";

        $sql_messages = "CREATE TABLE {$this->messages_table} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            ticket_id BIGINT(20) UNSIGNED NOT NULL,
            sender_type VARCHAR(20) NOT NULL, -- client / admin / ai
            sender_id BIGINT(20) UNSIGNED DEFAULT 0,
            message LONGTEXT NOT NULL,
            attachment_url TEXT,
            created_at DATETIME NOT NULL,
            PRIMARY KEY (id),
            KEY ticket_id (ticket_id),
            KEY sender_type (sender_type)
        ) $charset_collate;";

        dbDelta($sql_tickets);
        dbDelta($sql_messages);
    }

    /* -------------------- Network Admin Menu -------------------- */

    public function register_network_menu() {
        add_menu_page(
            __('AI Support Tickets', 'sitycloud'),
            __('AI Tickets', 'sitycloud'),
            'manage_network',
            'sity_ai_support_tickets',
            array($this, 'network_tickets_page'),
            'dashicons-sos',
            55
        );
    }

    /* -------------------- Admin Styles -------------------- */

    public function admin_styles($hook) {
        if ($hook !== 'toplevel_page_sity_ai_support_tickets-network') {
            // أحيانًا في الشبكة بيبقى الهوك مختلف، فنعمل check بسيط على page param
            if (!isset($_GET['page']) || $_GET['page'] !== 'sity_ai_support_tickets') {
                return;
            }
        }

        $custom_css = "
        .sity-admin-wrap {
            font-family: system-ui, -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
            max-width: 1100px;
        }
        .sity-admin-card {
            background: linear-gradient(135deg, rgba(124,58,237,0.08), rgba(255,255,255,0.9));
            border-radius: 18px;
            padding: 20px 24px;
            margin-bottom: 20px;
            border: 1px solid rgba(124,58,237,0.25);
            box-shadow: 0 12px 40px rgba(15,23,42,0.08);
            backdrop-filter: blur(16px);
        }
        .sity-ticket-badge {
            display:inline-block;
            padding:4px 10px;
            border-radius:999px;
            font-size:11px;
            font-weight:600;
        }
        .sity-status-open {background:#fef3c7;color:#92400e;}
        .sity-status-closed {background:#dcfce7;color:#14532d;}
        .sity-status-pending {background:#e0f2fe;color:#1d4ed8;}
        .sity-status-waiting {background:#fee2e2;color:#991b1b;}
        .sity-tag {
            font-size:11px;
            padding:3px 8px;
            border-radius:999px;
            border:1px solid rgba(148,163,184,0.6);
            color:#475569;
            margin-left:6px;
        }
        .sity-messages-timeline {
            border-left:2px solid rgba(148,163,184,0.5);
            margin-top:16px;
            padding-left:16px;
        }
        .sity-message-item {
            margin-bottom:16px;
            position:relative;
        }
        .sity-message-item::before {
            content:'';
            position:absolute;
            left:-11px;
            top:6px;
            width:10px;
            height:10px;
            border-radius:50%;
            background:#7c3aed;
            box-shadow:0 0 0 4px rgba(124,58,237,0.18);
        }
        .sity-message-meta {
            font-size:11px;
            color:#64748b;
            display:flex;
            gap:8px;
            margin-bottom:4px;
        }
        .sity-message-body {
            background:rgba(255,255,255,0.9);
            border-radius:12px;
            padding:10px 12px;
            border:1px solid rgba(148,163,184,0.35);
        }
        .sity-message-body.sity-from-client {border-color:rgba(34,197,94,0.4);}
        .sity-message-body.sity-from-ai {border-color:rgba(59,130,246,0.4);}
        .sity-message-body.sity-from-admin {border-color:rgba(124,58,237,0.5);}
        .sity-admin-table th, .sity-admin-table td {
            padding:8px 10px;
        }
        ";
        wp_add_inline_style('wp-admin', $custom_css);
    }

    /* -------------------- Network Admin Page -------------------- */

    public function network_tickets_page() {
        if (!current_user_can('manage_network')) {
            wp_die(__('You do not have permission.', 'sitycloud'));
        }

        global $wpdb;

        $view_ticket_id = isset($_GET['ticket_id']) ? intval($_GET['ticket_id']) : 0;

        echo '<div class="wrap sity-admin-wrap">';
        echo '<h1>AI Support Tickets</h1>';

        if ($view_ticket_id) {
            $ticket = $wpdb->get_row(
                $wpdb->prepare("SELECT * FROM {$this->tickets_table} WHERE id = %d", $view_ticket_id)
            );
            if (!$ticket) {
                echo '<div class="notice notice-error"><p>لم يتم العثور على التذكرة.</p></div>';
            } else {
                $this->render_admin_single_ticket($ticket);
            }
        } else {
            $this->render_admin_tickets_list();
        }

        echo '</div>';
    }

    private function render_admin_tickets_list() {
        global $wpdb;
        $tickets = $wpdb->get_results("SELECT * FROM {$this->tickets_table} ORDER BY updated_at DESC LIMIT 200");

        echo '<div class="sity-admin-card">';
        echo '<h2>جميع التذاكر في الشبكة</h2>';
        echo '<table class="widefat fixed striped sity-admin-table">';
        echo '<thead><tr>
                <th>ID</th>
                <th>الموقع</th>
                <th>المستخدم</th>
                <th>العنوان</th>
                <th>الحالة</th>
                <th>الأولوية</th>
                <th>قسم</th>
                <th>آخر تحديث</th>
                <th>إدارة</th>
            </tr></thead><tbody>';

        if ($tickets) {
            foreach ($tickets as $t) {
                $blog = get_blog_details($t->site_id);
                $user = get_user_by('id', $t->user_id);

                echo '<tr>';
                echo '<td>#' . esc_html($t->id) . '</td>';
                echo '<td>' . ($blog ? esc_html($blog->blogname) : 'غير معروف') . '</td>';
                echo '<td>' . ($user ? esc_html($user->user_login) : 'مستخدم #' . intval($t->user_id)) . '</td>';
                echo '<td>' . esc_html($t->subject) . '</td>';
                echo '<td>' . $this->render_status_badge($t->status) . '</td>';
                echo '<td>' . esc_html($t->priority) . '</td>';
                echo '<td>' . esc_html($t->department) . '</td>';
                echo '<td>' . esc_html($t->updated_at) . '</td>';
                echo '<td><a class="button button-primary" href="' . esc_url(network_admin_url('admin.php?page=sity_ai_support_tickets&ticket_id=' . $t->id)) . '">عرض التذكرة</a></td>';
                echo '</tr>';
            }
        } else {
            echo '<tr><td colspan="9">لا توجد تذاكر حتى الآن.</td></tr>';
        }

        echo '</tbody></table>';
        echo '</div>';
    }

    private function render_status_badge($status) {
        $class = 'sity-status-open';
        switch ($status) {
            case 'closed':
                $class = 'sity-status-closed';
                $label = 'مغلقة';
                break;
            case 'pending':
                $class = 'sity-status-pending';
                $label = 'بانتظار فريق الدعم';
                break;
            case 'waiting-client':
                $class = 'sity-status-waiting';
                $label = 'بانتظار رد العميل';
                break;
            default:
                $label = 'مفتوحة';
        }
        return '<span class="sity-ticket-badge ' . esc_attr($class) . '">' . esc_html($label) . '</span>';
    }

    private function render_admin_single_ticket($ticket) {
        global $wpdb;

        $messages = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$this->messages_table} WHERE ticket_id = %d ORDER BY created_at ASC",
                $ticket->id
            )
        );

        $blog = get_blog_details($ticket->site_id);
        $user = get_user_by('id', $ticket->user_id);

        echo '<div class="sity-admin-card">';
        echo '<h2>تذكرة #' . esc_html($ticket->id) . ' – ' . esc_html($ticket->subject) . '</h2>';

        echo '<p>';
        echo '<span class="sity-tag">الموقع: ' . ($blog ? esc_html($blog->blogname) : 'غير معروف') . '</span>';
        echo '<span class="sity-tag">المستخدم: ' . ($user ? esc_html($user->user_login) : 'User #' . intval($ticket->user_id)) . '</span>';
        echo '<span class="sity-tag">القسم: ' . esc_html($ticket->department) . '</span>';
        echo '<span class="sity-tag">الأولوية: ' . esc_html($ticket->priority) . '</span>';
        echo '</p>';

        echo '<p>الحالة الحالية: ' . $this->render_status_badge($ticket->status) . '</p>';

        echo '<div class="sity-messages-timeline">';
        if ($messages) {
            foreach ($messages as $m) {
                $cls = 'sity-from-client';
                $label = 'العميل';
                if ($m->sender_type === 'admin') {
                    $cls = 'sity-from-admin';
                    $label = 'الأدمن';
                } elseif ($m->sender_type === 'ai') {
                    $cls = 'sity-from-ai';
                    $label = 'الذكاء الاصطناعي';
                }

                echo '<div class="sity-message-item">';
                echo '<div class="sity-message-meta">';
                echo '<span>' . esc_html($label) . '</span>';
                echo '<span>•</span>';
                echo '<span>' . esc_html($m->created_at) . '</span>';
                echo '</div>';
                echo '<div class="sity-message-body ' . esc_attr($cls) . '">';
                echo wpautop(esc_html($m->message));
                if (!empty($m->attachment_url)) {
                    echo '<p><strong>مرفق:</strong> <a href="' . esc_url($m->attachment_url) . '" target="_blank">تحميل / عرض</a></p>';
                }
                echo '</div>';
                echo '</div>';
            }
        } else {
            echo '<p>لا توجد رسائل بعد.</p>';
        }
        echo '</div>';

        echo '<hr />';
        echo '<h3>إضافة رد جديد</h3>';

        echo '<form method="post" enctype="multipart/form-data">';
        wp_nonce_field('sity_admin_reply_ticket', 'sity_admin_reply_nonce');
        echo '<input type="hidden" name="sity_action" value="admin_reply_ticket" />';
        echo '<input type="hidden" name="ticket_id" value="' . esc_attr($ticket->id) . '" />';

        echo '<p><label for="sity_reply_message">الرد:</label><br />';
        echo '<textarea id="sity_reply_message" name="reply_message" rows="5" style="width:100%;" required></textarea></p>';

        echo '<p><label for="sity_reply_status">تحديث الحالة:</label><br />';
        echo '<select id="sity_reply_status" name="ticket_status">
                <option value="open" ' . selected($ticket->status, 'open', false) . '>مفتوحة</option>
                <option value="pending" ' . selected($ticket->status, 'pending', false) . '>بانتظار فريق الدعم</option>
                <option value="waiting-client" ' . selected($ticket->status, 'waiting-client', false) . '>بانتظار رد العميل</option>
                <option value="closed" ' . selected($ticket->status, 'closed', false) . '>مغلقة</option>
            </select></p>';

        echo '<p><label for="sity_reply_file">إرفاق ملف (اختياري):</label><br />';
        echo '<input type="file" id="sity_reply_file" name="reply_file" /></p>';

        submit_button('إرسال الرد وتحديث التذكرة');

        echo '</form>';

        echo '</div>';
    }

    /* -------------------- Handle Admin POST (reply ticket) -------------------- */

    public function handle_admin_post_requests() {
        if (!is_network_admin()) {
            return;
        }
        if (!isset($_POST['sity_action']) || $_POST['sity_action'] !== 'admin_reply_ticket') {
            return;
        }
        if (!current_user_can('manage_network')) {
            return;
        }
        if (!isset($_POST['sity_admin_reply_nonce']) || !wp_verify_nonce($_POST['sity_admin_reply_nonce'], 'sity_admin_reply_ticket')) {
            return;
        }

        global $wpdb;

        $ticket_id = isset($_POST['ticket_id']) ? intval($_POST['ticket_id']) : 0;
        $message   = isset($_POST['reply_message']) ? wp_kses_post($_POST['reply_message']) : '';
        $status    = isset($_POST['ticket_status']) ? sanitize_text_field($_POST['ticket_status']) : 'open';

        if (!$ticket_id || empty($message)) {
            return;
        }

        $ticket = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM {$this->tickets_table} WHERE id = %d", $ticket_id)
        );
        if (!$ticket) {
            return;
        }

        // معالجة المرفق إن وجد
        $attachment_url = '';
        if (!empty($_FILES['reply_file']['name'])) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            $uploaded = wp_handle_upload($_FILES['reply_file'], array('test_form' => false));
            if (!isset($uploaded['error']) && isset($uploaded['url'])) {
                $attachment_url = $uploaded['url'];
            }
        }

        $now = current_time('mysql');

        $wpdb->insert(
            $this->messages_table,
            array(
                'ticket_id'      => $ticket_id,
                'sender_type'    => 'admin',
                'sender_id'      => get_current_user_id(),
                'message'        => $message,
                'attachment_url' => $attachment_url,
                'created_at'     => $now,
            ),
            array('%d', '%s', '%d', '%s', '%s', '%s')
        );

        $wpdb->update(
            $this->tickets_table,
            array(
                'status'        => $status,
                'updated_at'    => $now,
                'last_reply_by' => 'admin',
                'needs_human'   => 0,
            ),
            array('id' => $ticket_id),
            array('%s', '%s', '%s', '%d'),
            array('%d')
        );

        wp_safe_redirect(network_admin_url('admin.php?page=sity_ai_support_tickets&ticket_id=' . $ticket_id . '&updated=1'));
        exit;
    }

    /* -------------------- Front-end Shortcode -------------------- */

    public function shortcode_render($atts) {
        if (!is_user_logged_in()) {
            return '<div class="sity-support-guest">يجب تسجيل الدخول لعرض وإنشاء تذاكر الدعم.</div>';
        }

        global $wpdb;
        $current_user = wp_get_current_user();
        $current_site = get_current_blog_id();

        // اختيار تذكرة واحدة لو فيه پارام
        $view_ticket_id = isset($_GET['sity_ticket']) ? intval($_GET['sity_ticket']) : 0;
        $ticket = null;
        $messages = array();

        if ($view_ticket_id) {
            $ticket = $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT * FROM {$this->tickets_table} WHERE id = %d AND user_id = %d AND site_id = %d",
                    $view_ticket_id,
                    $current_user->ID,
                    $current_site
                )
            );
            if ($ticket) {
                $messages = $wpdb->get_results(
                    $wpdb->prepare(
                        "SELECT * FROM {$this->messages_table} WHERE ticket_id = %d ORDER BY created_at ASC",
                        $ticket->id
                    )
                );
            }
        }

        // جلب جميع تذاكر هذا العميل في هذا الموقع
        $tickets = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$this->tickets_table} WHERE user_id = %d AND site_id = %d ORDER BY updated_at DESC",
                $current_user->ID,
                $current_site
            )
        );

        ob_start();
        ?>

        <style>
        .sity-support-wrapper {
            direction: rtl;
            font-family: system-ui, -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
            max-width: 1100px;
            margin: 30px auto;
            padding: 10px;
        }
        .sity-support-grid {
            display: grid;
            grid-template-columns: minmax(0, 2fr) minmax(0, 3fr);
            gap: 20px;
        }
        @media (max-width: 900px) {
            .sity-support-grid {
                grid-template-columns: minmax(0, 1fr);
            }
        }
        .sity-card {
            background: radial-gradient(circle at top left, rgba(124,58,237,0.18), rgba(255,255,255,0.95));
            border-radius: 22px;
            padding: 20px 22px;
            border: 1px solid rgba(148,163,184,0.4);
            box-shadow: 0 18px 45px rgba(15,23,42,0.12);
            backdrop-filter: blur(18px);
        }
        .sity-card-header {
            display:flex;
            justify-content:space-between;
            align-items:center;
            margin-bottom:12px;
        }
        .sity-card-title {
            font-size:18px;
            font-weight:700;
            color:#0f172a;
        }
        .sity-pill {
            font-size:11px;
            padding:4px 10px;
            border-radius:999px;
            background:rgba(124,58,237,0.08);
            border:1px solid rgba(124,58,237,0.3);
            color:#7c3aed;
        }

        .sity-input, .sity-select, .sity-textarea {
            width:100%;
            border-radius:14px;
            border:1px solid rgba(148,163,184,0.6);
            padding:9px 11px;
            font-size:13px;
            margin-bottom:10px;
            background:rgba(255,255,255,0.9);
            outline:none;
            transition:all .15s ease;
        }
        .sity-input:focus, .sity-select:focus, .sity-textarea:focus {
            border-color:#7c3aed;
            box-shadow:0 0 0 1px rgba(124,58,237,0.4);
        }
        .sity-label {
            font-size:13px;
            font-weight:600;
            margin-bottom:4px;
            color:#1f2933;
        }
        .sity-button-primary {
            display:inline-flex;
            align-items:center;
            justify-content:center;
            gap:6px;
            border-radius:999px;
            border:none;
            padding:9px 18px;
            font-size:13px;
            font-weight:600;
            background:linear-gradient(135deg,#8b5cf6,#6366f1);
            color:white;
            cursor:pointer;
            box-shadow:0 12px 25px rgba(99,102,241,0.45);
            transition:transform .12s ease, box-shadow .12s ease, filter .12s ease;
        }
        .sity-button-primary:hover {
            transform:translateY(-1px);
            box-shadow:0 16px 35px rgba(99,102,241,0.55);
            filter:brightness(1.05);
        }
        .sity-ticket-list {
            max-height:420px;
            overflow:auto;
            margin-top:10px;
        }
        .sity-ticket-row {
            display:flex;
            justify-content:space-between;
            align-items:flex-start;
            padding:9px 10px;
            border-radius:14px;
            background:rgba(255,255,255,0.9);
            border:1px solid rgba(226,232,240,0.9);
            margin-bottom:7px;
            cursor:pointer;
            transition:all .12s ease;
        }
        .sity-ticket-row:hover {
            border-color:rgba(124,58,237,0.6);
            box-shadow:0 6px 20px rgba(15,23,42,0.12);
        }
        .sity-ticket-main {
            flex:1;
        }
        .sity-ticket-subject {
            font-size:14px;
            font-weight:600;
            color:#111827;
            margin-bottom:2px;
        }
        .sity-ticket-meta {
            font-size:11px;
            color:#6b7280;
        }
        .sity-ticket-status {
            font-size:11px;
            padding:3px 9px;
            border-radius:999px;
            border:1px solid rgba(148,163,184,0.7);
            background:rgba(248,250,252,0.9);
            color:#4b5563;
            margin-bottom:4px;
            text-align:center;
        }
        .sity-status-open {background:#fef3c7;color:#92400e;border-color:#fbbf24;}
        .sity-status-closed {background:#dcfce7;color:#14532d;border-color:#22c55e;}
        .sity-status-pending {background:#e0f2fe;color:#1d4ed8;border-color:#3b82f6;}
        .sity-status-waiting-client {background:#fee2e2;color:#991b1b;border-color:#f97373;}

        .sity-ticket-view {
            max-height:460px;
            overflow:auto;
            margin-top:4px;
        }
        .sity-message-item {
            margin-bottom:12px;
        }
        .sity-message-meta {
            font-size:11px;
            color:#64748b;
            margin-bottom:3px;
        }
        .sity-message-bubble {
            border-radius:16px;
            padding:9px 11px;
            font-size:13px;
            border:1px solid rgba(148,163,184,0.5);
            background:rgba(255,255,255,0.96);
        }
        .sity-message-client {border-color:rgba(34,197,94,0.55);}
        .sity-message-ai {border-color:rgba(59,130,246,0.55);background:linear-gradient(135deg,rgba(59,130,246,0.08),rgba(255,255,255,0.96));}
        .sity-message-admin {border-color:rgba(124,58,237,0.7);background:linear-gradient(135deg,rgba(124,58,237,0.08),rgba(255,255,255,0.96));}

        .sity-attachment {
            margin-top:5px;
            font-size:11px;
        }
        .sity-attachment a {
            color:#7c3aed;
            text-decoration:none;
        }
        .sity-attachment a:hover {
            text-decoration:underline;
        }

        .sity-support-guest {
            direction:rtl;
            background:rgba(15,23,42,0.85);
            color:white;
            padding:16px 18px;
            border-radius:18px;
            text-align:center;
        }
        </style>

        <div class="sity-support-wrapper">
            <div class="sity-support-grid">

                <!-- إنشاء تذكرة جديدة -->
                <div class="sity-card">
                    <div class="sity-card-header">
                        <div class="sity-card-title">فتح تذكرة دعم جديدة</div>
                        <div class="sity-pill">يتم تسجيل كل شيء داخل الشبكة</div>
                    </div>
                    <form method="post" enctype="multipart/form-data">
                        <?php wp_nonce_field('sity_new_ticket', 'sity_new_ticket_nonce'); ?>
                        <input type="hidden" name="sity_action" value="new_ticket" />

                        <div>
                            <div class="sity-label">عنوان التذكرة</div>
                            <input type="text" name="ticket_subject" class="sity-input" required placeholder="مثال: مشكلة في تسجيل الدخول لموقعي" />
                        </div>

                        <div>
                            <div class="sity-label">القسم</div>
                            <select name="ticket_department" class="sity-select">
                                <option value="technical">الدعم التقني</option>
                                <option value="billing">الفواتير والمدفوعات</option>
                                <option value="accounts">الحسابات والصلاحيات</option>
                                <option value="general">استفسار عام</option>
                            </select>
                        </div>

                        <div>
                            <div class="sity-label">الأولوية</div>
                            <select name="ticket_priority" class="sity-select">
                                <option value="normal">عادية</option>
                                <option value="high">مرتفعة</option>
                                <option value="urgent">عاجلة</option>
                            </select>
                        </div>

                        <div>
                            <div class="sity-label">وصف المشكلة / الطلب</div>
                            <textarea name="ticket_message" class="sity-textarea" rows="5" required placeholder="اشرح المشكلة أو طلبك بالتفصيل قدر الإمكان..."></textarea>
                        </div>

                        <div>
                            <div class="sity-label">مرفق (اختياري)</div>
                            <input type="file" name="ticket_file" />
                        </div>

                        <button type="submit" class="sity-button-primary">
                            إرسال التذكرة
                            <span>↗</span>
                        </button>
                    </form>
                </div>

                <!-- قائمة التذاكر + عرض تذكرة -->
                <div class="sity-card">
                    <div class="sity-card-header">
                        <div class="sity-card-title">تذاكر الدعم الخاصة بك</div>
                        <div class="sity-pill">سجل كامل للمحادثات</div>
                    </div>

                    <div class="sity-ticket-list">
                        <?php
                        if ($tickets) {
                            foreach ($tickets as $t) {
                                $status_class = 'sity-status-open';
                                $status_label = 'مفتوحة';
                                switch ($t->status) {
                                    case 'closed':
                                        $status_class = 'sity-status-closed';
                                        $status_label = 'مغلقة';
                                        break;
                                    case 'pending':
                                        $status_class = 'sity-status-pending';
                                        $status_label = 'بانتظار فريق الدعم';
                                        break;
                                    case 'waiting-client':
                                        $status_class = 'sity-status-waiting-client';
                                        $status_label = 'بانتظار ردك';
                                        break;
                                }

                                $ticket_url = add_query_arg('sity_ticket', $t->id, get_permalink());

                                echo '<div class="sity-ticket-row" onclick="window.location.href=\'' . esc_url($ticket_url) . '\'">';
                                echo '<div class="sity-ticket-main">';
                                echo '<div class="sity-ticket-subject">#' . esc_html($t->id) . ' – ' . esc_html($t->subject) . '</div>';
                                echo '<div class="sity-ticket-meta">';
                                echo 'القسم: ' . esc_html($t->department) . ' • ';
                                echo 'الأولوية: ' . esc_html($t->priority) . '<br />';
                                echo 'آخر تحديث: ' . esc_html($t->updated_at);
                                echo '</div>';
                                echo '</div>';
                                echo '<div>';
                                echo '<div class="sity-ticket-status ' . esc_attr($status_class) . '">' . esc_html($status_label) . '</div>';
                                echo '</div>';
                                echo '</div>';
                            }
                        } else {
                            echo '<p>لا توجد تذاكر حتى الآن. ابدأ بفتح تذكرة جديدة من النموذج على اليسار.</p>';
                        }
                        ?>
                    </div>

                    <?php if ($ticket) : ?>
                        <hr style="margin:18px 0;opacity:0.4;" />
                        <div class="sity-ticket-view">
                            <h3 style="font-size:15px;margin-bottom:8px;">محادثة التذكرة #<?php echo esc_html($ticket->id); ?></h3>
                            <?php
                            if ($messages) {
                                foreach ($messages as $m) {
                                    $label = 'أنت';
                                    $cls = 'sity-message-client';
                                    if ($m->sender_type === 'admin') {
                                        $label = 'فريق الدعم';
                                        $cls = 'sity-message-admin';
                                    } elseif ($m->sender_type === 'ai') {
                                        $label = 'الرد الذكي (AI)';
                                        $cls = 'sity-message-ai';
                                    }
                                    ?>
                                    <div class="sity-message-item">
                                        <div class="sity-message-meta">
                                            <?php echo esc_html($label); ?> • <?php echo esc_html($m->created_at); ?>
                                        </div>
                                        <div class="sity-message-bubble <?php echo esc_attr($cls); ?>">
                                            <?php echo wpautop(esc_html($m->message)); ?>
                                            <?php if (!empty($m->attachment_url)) : ?>
                                                <div class="sity-attachment">
                                                    <strong>مرفق:</strong>
                                                    <a href="<?php echo esc_url($m->attachment_url); ?>" target="_blank">تحميل / عرض</a>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <?php
                                }
                            } else {
                                echo '<p>لا توجد رسائل بعد لهذه التذكرة.</p>';
                            }
                            ?>

                            <?php if ($ticket->status !== 'closed') : ?>
                                <form method="post" enctype="multipart/form-data" style="margin-top:14px;">
                                    <?php wp_nonce_field('sity_reply_ticket', 'sity_reply_ticket_nonce'); ?>
                                    <input type="hidden" name="sity_action" value="client_reply_ticket" />
                                    <input type="hidden" name="ticket_id" value="<?php echo esc_attr($ticket->id); ?>" />

                                    <div>
                                        <div class="sity-label">إضافة رد جديد</div>
                                        <textarea name="reply_message" class="sity-textarea" rows="4" required></textarea>
                                    </div>

                                    <div>
                                        <div class="sity-label">مرفق (اختياري)</div>
                                        <input type="file" name="reply_file" />
                                    </div>

                                    <button type="submit" class="sity-button-primary">
                                        إرسال الرد
                                        <span>💬</span>
                                    </button>
                                </form>
                            <?php else : ?>
                                <p style="font-size:12px;color:#6b7280;margin-top:10px;">هذه التذكرة مغلقة. يمكنك فتح تذكرة جديدة إذا لديك استفسار آخر.</p>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>

            </div>
        </div>

        <?php
        return ob_get_clean();
    }

    /* -------------------- Front-end POST Handlers -------------------- */

    public function handle_front_post_requests() {
        if (!is_user_logged_in()) {
            return;
        }

        if (!isset($_POST['sity_action'])) {
            return;
        }

        $action = sanitize_text_field($_POST['sity_action']);

        if ($action === 'new_ticket') {
            $this->handle_new_ticket();
        } elseif ($action === 'client_reply_ticket') {
            $this->handle_client_reply();
        }
    }

    private function handle_new_ticket() {
        if (!isset($_POST['sity_new_ticket_nonce']) || !wp_verify_nonce($_POST['sity_new_ticket_nonce'], 'sity_new_ticket')) {
            return;
        }

        global $wpdb;
        $current_user = wp_get_current_user();
        $current_site = get_current_blog_id();

        $subject    = isset($_POST['ticket_subject']) ? sanitize_text_field($_POST['ticket_subject']) : '';
        $department = isset($_POST['ticket_department']) ? sanitize_text_field($_POST['ticket_department']) : 'general';
        $priority   = isset($_POST['ticket_priority']) ? sanitize_text_field($_POST['ticket_priority']) : 'normal';
        $message    = isset($_POST['ticket_message']) ? wp_kses_post($_POST['ticket_message']) : '';

        if (empty($subject) || empty($message)) {
            return;
        }

        // مرفق العميل
        $attachment_url = '';
        if (!empty($_FILES['ticket_file']['name'])) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            $uploaded = wp_handle_upload($_FILES['ticket_file'], array('test_form' => false));
            if (!isset($uploaded['error']) && isset($uploaded['url'])) {
                $attachment_url = $uploaded['url'];
            }
        }

        $now = current_time('mysql');

        // إنشاء التذكرة
        $wpdb->insert(
            $this->tickets_table,
            array(
                'site_id'      => $current_site,
                'user_id'      => $current_user->ID,
                'subject'      => $subject,
                'department'   => $department,
                'priority'     => $priority,
                'status'       => 'open',
                'needs_human'  => 0,
                'created_at'   => $now,
                'updated_at'   => $now,
                'last_reply_by'=> 'client',
            ),
            array('%d', '%d', '%s', '%s', '%s', '%s', '%d', '%s', '%s', '%s')
        );

        $ticket_id = $wpdb->insert_id;

        if (!$ticket_id) {
            return;
        }

        // أول رسالة من العميل
        $wpdb->insert(
            $this->messages_table,
            array(
                'ticket_id'      => $ticket_id,
                'sender_type'    => 'client',
                'sender_id'      => $current_user->ID,
                'message'        => $message,
                'attachment_url' => $attachment_url,
                'created_at'     => $now,
            ),
            array('%d', '%s', '%d', '%s', '%s', '%s')
        );

        // رد تلقائي من AI (تقدر تربطه بأي API)
        $ai_reply = $this->generate_ai_reply($subject, $message, $department, $priority);

        if ($ai_reply) {
            $wpdb->insert(
                $this->messages_table,
                array(
                    'ticket_id'      => $ticket_id,
                    'sender_type'    => 'ai',
                    'sender_id'      => 0,
                    'message'        => $ai_reply,
                    'attachment_url' => '',
                    'created_at'     => current_time('mysql'),
                ),
                array('%d', '%s', '%d', '%s', '%s', '%s')
            );

            $wpdb->update(
                $this->tickets_table,
                array(
                    'updated_at'    => current_time('mysql'),
                    'last_reply_by' => 'ai',
                    'status'        => 'pending', // مثلاً: بانتظار متابعة من فريق الدعم بعد رد AI
                    'needs_human'   => 1, // علشان تشوفها في لوحة الشبكة وتتابع لو محتاج
                ),
                array('id' => $ticket_id),
                array('%s', '%s', '%s', '%d'),
                array('%d')
            );
        } else {
            // فشل AI → نعلّم أن التذكرة تحتاج تدخل بشري
            $wpdb->update(
                $this->tickets_table,
                array(
                    'needs_human' => 1,
                ),
                array('id' => $ticket_id),
                array('%d'),
                array('%d')
            );
        }

        // رجّع العميل لنفس الصفحة مع تذكرة مفتوحة
        $redirect = add_query_arg('sity_ticket', $ticket_id, wp_get_referer() ?: home_url());
        wp_safe_redirect($redirect);
        exit;
    }

    private function handle_client_reply() {
        if (!isset($_POST['sity_reply_ticket_nonce']) || !wp_verify_nonce($_POST['sity_reply_ticket_nonce'], 'sity_reply_ticket')) {
            return;
        }

        global $wpdb;
        $current_user = wp_get_current_user();
        $current_site = get_current_blog_id();

        $ticket_id = isset($_POST['ticket_id']) ? intval($_POST['ticket_id']) : 0;
        $message   = isset($_POST['reply_message']) ? wp_kses_post($_POST['reply_message']) : '';

        if (!$ticket_id || empty($message)) {
            return;
        }

        $ticket = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM {$this->tickets_table} WHERE id = %d AND user_id = %d AND site_id = %d",
                $ticket_id,
                $current_user->ID,
                $current_site
            )
        );
        if (!$ticket) {
            return;
        }

        // مرفق من العميل في الرد
        $attachment_url = '';
        if (!empty($_FILES['reply_file']['name'])) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            $uploaded = wp_handle_upload($_FILES['reply_file'], array('test_form' => false));
            if (!isset($uploaded['error']) && isset($uploaded['url'])) {
                $attachment_url = $uploaded['url'];
            }
        }

        $now = current_time('mysql');

        $wpdb->insert(
            $this->messages_table,
            array(
                'ticket_id'      => $ticket_id,
                'sender_type'    => 'client',
                'sender_id'      => $current_user->ID,
                'message'        => $message,
                'attachment_url' => $attachment_url,
                'created_at'     => $now,
            ),
            array('%d', '%s', '%d', '%s', '%s', '%s')
        );

        // تحديث حالة التذكرة: بانتظار فريق الدعم الآن
        $wpdb->update(
            $this->tickets_table,
            array(
                'status'        => 'pending',
                'updated_at'    => $now,
                'last_reply_by' => 'client',
                'needs_human'   => 1,
            ),
            array('id' => $ticket_id),
            array('%s', '%s', '%s', '%d'),
            array('%d')
        );

        $redirect = add_query_arg('sity_ticket', $ticket_id, wp_get_referer() ?: home_url());
        wp_safe_redirect($redirect);
        exit;
    }

    /* -------------------- AI Reply Stub -------------------- */

    /**
     * هنا قلب الذكاء الاصطناعي:
     * حالياً بيرجع رد تلقائي ذكي ثابت.
     * تقدر تستبدل محتوى الفنكشن دي باتصال حقيقي بـ Gemini / OpenAI / أي مزوّد.
     */
    private function generate_ai_reply($subject, $message, $department, $priority) {
        // مثال: لو حابب توصلها ب API خارجي
        // 1) جهّز البرومبت
        // 2) استعمل wp_remote_post للاتصال
        // 3) رجّع نص الرد أو false لو فشل

        // رد افتراضي بسيط لحد ما تربطه بالـ API الحقيقي
        $reply  = "شكرًا لفتح تذكرة الدعم 👋\n\n";
        $reply .= "تم استلام طلبك بخصوص: \"" . trim($subject) . "\"\n";
        $reply .= "سنراجع التفاصيل ونعود إليك في أقرب وقت ممكن.\n\n";
        $reply .= "هذا الرد الآلي يساعدك تتأكد إن تذكرتك دخلت النظام بنجاح ضمن شبكة Sity Cloud.";

        return $reply;
    }

}

SityCloud_AI_Support_Tickets::instance();
