<?php

class Analytify_Authors extends WP_Analytify_Pro_Base {

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 */
	protected $plugin_name = 'wp-analytify-authors';

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {

		parent::__construct();

		$this->setup_constants();
		$this->set_locale();
		$this->hooks();
	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function hooks() {

		add_action( 'admin_enqueue_scripts', array( $this, 'authors_admin_scripts' ), 10, 1 );
		add_action( 'analytify_add_submenu' , array( $this, 'add_menu' ), 45 );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
		add_action( 'admin_init', array( $this, 'wp_analytify_authors_register_option' ) );
		add_action( 'admin_init', array( $this, '_plugin_updater' ), 0 );
		add_action( 'wp_analytify_authors_license_key', array( $this, 'display_license_form' ), 11 );
		add_action( 'wp_ajax_wpanalytify_authors_activate_license', array( $this, 'ajax_activate_license' ) );
		add_action( 'analytify_dashboad_dropdown_option' , array( $this, 'dashboad_dropdown_option' ) );
	}

	/**
	 * Define constant if not already set
	 *
	 * @param  string      $name
	 * @param  string|bool $value
	 */
	private function define( $name, $value ) {
		if ( ! defined( $name ) ) {
			define( $name, $value );
		}
	}

	/**
	 * Setup plugin constants
	 *
	 * @access      private
	 * @since       1.0.0
	 * @return      void
	 */
	private function setup_constants() {

		$this->define( 'AUTHORS_ANALYTIFY_PLUGIN_FILE', __FILE__ );
		$this->define( 'AUTHORS_ANALYTIFY_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
		$this->define( 'AUTHORS_ANALYTIFY_PLUGIN_DIR', dirname( __FILE__ ) );
		$this->define( 'AUTHORS_ANALYTIFY_PRODUCT_NAME', 'Analytify - Authors Tracking Addon' );
		$this->define( 'AUTHORS_ANALYTIFY_PRODUCT_ID', 229554 );
		$this->define( 'AUTHORS_ANALYTIFY_STORE_URL', 'https://analytify.io' );
	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Plugin_Name_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {
		$plugin_dir = basename( dirname( ANALYTIFY_AUTHORS_ROOT_PATH ) );
		load_plugin_textdomain( 'wp-analytify-authors', false, $plugin_dir . '/languages/' );
	}

	/**
	 * Load Admin Scripts
	 *
	 * @since 1.0
	 */
	public function admin_scripts( $page ) {
		if ( 'analytify_page_analytify-settings' == $page ) {
			wp_enqueue_script( 'analytify_authors_main_js', plugins_url( 'assets/js/main.js', __DIR__ ), array( 'jquery' ), ANALYTIFY_AUTHORS_DASHBOARD_VERSION );
		}
	}

	/**
	 * Plugin row meta links
	 * 
	 * @param array  $input already defined meta links
	 * @param string $file plugin file path and name being processed
	 * @return array $input
	 */
	public function plugin_row_meta( $input, $file ) {

		// Exit if its other plugins.
		if ( 'wp-analytify-forms/wp-analytify-forms.php' !== $file ) { return $input; }

		// Remove the 'view details' org link.
		unset( $input[2] );

		// Add the plugin site link.
		$input[] = '<a href="https://analytify.io/add-ons/forms-tracking/" target="_blank">Visit plugin site</a>';

		return $input;
	}

	/**
	 * Creates our settings in the options table.
	 *
	 * @return void
	 */
	public function wp_analytify_authors_register_option() {

		register_setting( 'analytify-settings', 'analytify_authors_license_key', array( $this, 'analytify_authors_sanitize_license' ) );
	}

	/**
	 * Sanitize the license key
	 *
	 * @param string $new
	 * 
	 * @return string
	 */
	public function analytify_authors_sanitize_license( $new ) {

		$old = get_option( 'analytify_authors_license_key' );

		if ( $old && $old != $new ) {
			delete_option( 'analytify_authors_license_status' ); // new license has been entered, so must reactivate
		}
		return $new;

	}

	/**
	 * Auto update addon.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function _plugin_updater() {

		// Retrieve our license key from the DB.
		$wpa_license_key = trim( get_option( 'analytify_authors_license_key' ) );

		// Setup the updater.
		if ( class_exists( 'ANALYTIFY_SL_Plugin_Updater' ) ) {
			$edd_updater = new ANALYTIFY_SL_Plugin_Updater(
				AUTHORS_ANALYTIFY_STORE_URL, 'wp-analytify-authors/wp-analytify-authors.php', array(
					'version' => ANALYTIFY_AUTHORS_DASHBOARD_VERSION, // current version number
					'license' => $wpa_license_key, // license key (used get_option above to retrieve from DB)
					'item_id' => AUTHORS_ANALYTIFY_PRODUCT_ID, // ID of this plugin
					'author'  => 'Muhammad Adnan', // author of this plugin
				)
			);
		}
	}

  	/**
	 * Enqueue admin scripts.
	 *
	 * @since     1.0.0
	 * @return    void
	 */
  	public function authors_admin_scripts( $location ) {

		if ( 'analytify_page_analytify-authors' !== $location ) {
			return;
   		}

    	wp_enqueue_style( 'analytify_author_css', plugins_url('/assets/css/main.css', __DIR__), false );
		wp_enqueue_style( 'analytify-dashboard-authors', plugins_url( 'assets/css/common-dashboard.css', dirname( __FILE__ ) ), array(), ANALYTIFY_AUTHORS_DASHBOARD_VERSION );

		wp_enqueue_script( 'analytify-stats-authors', plugins_url( 'assets/js/stats-authors.js', dirname( __FILE__ ) ), array( 'jquery' ), ANALYTIFY_AUTHORS_DASHBOARD_VERSION, true );
		wp_localize_script(
			'analytify-stats-authors',
			'analytify_stats_authors',
			array(
				'url'              => esc_url_raw( rest_url( 'wp-analytify/v1/get_author_report/' ) ),
				'nonce'            => wp_create_nonce( 'wp_rest' ),
				'delimiter'        => is_callable( array('WPANALYTIFY_Utils', 'get_delimiter') ) ? WPANALYTIFY_Utils::get_delimiter() : '?',
				'load_via_ajax'    => true,
				'no_stats_message' => __( 'No activity during this period.', 'wp-analytify-authors' ),
				'error_message'    => __( 'Something went wrong. Please try again.', 'wp-analytify-authors' ),
			),
		);
  	}

	/**
	 * Add dropdown option for admin dashboard.
	 *
	 * @since 1.0.0
	 *
	 */
	public function dashboad_dropdown_option() {

		echo  '<li><a href="'. admin_url( 'admin.php?page=analytify-authors' ) .'">Authors</a></li>';
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {

		return $this->plugin_name;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {

		return $this->version;
	}

	/**
	 * Add menu for Author Tracking Dashboard page
	 */
	public function add_menu() {

		// add_submenu_page( $parent_slug, $page_title, $menu_title, $capability, $menu_slug, $function = '' )
		add_submenu_page( 'analytify-dashboard', esc_html__( 'Authors Tracking', 'wp-analytify-authors' ), esc_html__( 'Authors Tracking', 'wp-analytify-authors' ), 'edit_posts', 'analytify-authors', array( __CLASS__, 'authors_dashboard_page' ), 35 );
	}

	/**
	 * Output Analytify for Authors license activation key fields/inputs.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function display_license_form() {

		$license_authors = get_option( 'analytify_authors_license_key' );
		$status_authors  = get_option( 'analytify_authors_license_status' );
		?>

		<tr valign="top">
			<th scope="row" valign="top">
				<?php esc_html_e( 'Authors Tracking:', 'wp-analytify-authors' ); ?>
			</th>
			<?php if ( 'valid' === $status_authors ) : ?>
			<td class="authors-license-row">
				<?php echo $this->get_formatted_masked_authors_license( $license_authors ); ?>
			</td>
			<?php else : ?>
			<td class="authors-license-row">
				<input id="analytify_authors_license_key" name="analytify_authors_license_key" type="text" class="regular-text" value="<?php esc_attr_e( $license_authors ); ?>" />
				<input type="submit" class="button-secondary" id="analytify_authors_license_activate" name="analytify_license_activate" value="<?php esc_html_e( 'Activate License', 'wp-analytify-authors' ); ?>"/>
				<br /><p id="authors-license-status">
				<?php
				if ( $status_authors ) {
					echo $status_authors;
				}
				?>
				</p>
			</td>
			<?php
		 	endif;
			?>
		  </tr>
		<?php
	}		

	/**
	 * Authors dashboard page.
	 *
	 * @return void
	 */
	public static function authors_dashboard_page() {

		$screen = get_current_screen();

		if ( strpos( $screen->base, 'analytify-authors' ) !== false ) {

			$wp_analytify   = $GLOBALS['WP_ANALYTIFY'];
			$selected_stats = $wp_analytify->settings->get_option( 'show_analytics_panels_dashboard', 'wp-analytify-dashboard', array() );

			$dashboard_profile_id = is_callable( array('WPANALYTIFY_Utils', 'get_reporting_property') ) ? WPANALYTIFY_Utils::get_reporting_property() : '';
			$access_token         = get_option( 'post_analytics_token' );
			$version              = defined( 'ANALYTIFY_PRO_VERSION' ) ? ANALYTIFY_PRO_VERSION : ANALYTIFY_VERSION;

			/**
			 * Default dates.
			 * $_POST dates are checked incase the Per version is older than 5.0.0.
			 */
			$start_date = isset( $_POST['analytify_date_start'] ) ? sanitize_text_field( wp_unslash( $_POST['analytify_date_start'] ) ) : wp_date( 'Y-m-d', strtotime( '-1 month' ) );
			$end_date   = isset( $_POST['analytify_date_end'] ) ? sanitize_text_field( wp_unslash( $_POST['analytify_date_end'] ) ) : wp_date( 'Y-m-d', strtotime( 'now' ) );

			// Get compare dates for legacy version (before v5.0.0).
			$date_diff          = is_callable( array('WPANALYTIFY_Utils', 'calculate_date_diff') ) ? WPANALYTIFY_Utils::calculate_date_diff( $start_date, $end_date ) : array('start_date' => '', 'end_date' => '');
			$compare_start_date = $date_diff['start_date'];
			$compare_end_date   = $date_diff['end_date'];

			/*
			 * Check with roles assigned at dashboard settings.
			 */
			$is_access_level = $wp_analytify->settings->get_option( 'show_analytics_roles_dashboard', 'wp-analytify-dashboard' );

			// Show dashboard to admin incase of empty access roles.
			if ( empty( $is_access_level ) ) {
				$is_access_level = array( 'administrator' );
			}

			if ( $wp_analytify->pa_check_roles( $is_access_level ) ) {
				if ( $access_token ) {

					// Dequeue event calendar js.
					wp_dequeue_script( 'tribe-common' );

					$current_user = wp_get_current_user();

					$users = get_users(
						array(
							'orderby'  => 'display_name',
							'order'    => 'ASC',
							'role__in' => apply_filters( 'analytify_authors_dashboard_stat_roles', array( 'administrator', 'author', 'editor' ) ),
						)
					);

					require_once ANALYTIFY_AUTHORS_ROOT_PATH . '/views/author-dashboard.php';
				} else {
					_e( 'You must be authenticated to see the Analytics Dashboard.', 'wp-analytify-authors' );
				}
			} else {
				_e( 'You don\'t have access to Analytify Dashboard.', 'wp-analytify-authors' );
			}
		}
	}

	/**
	 *
	 * Send Ajax on Active license.
	 *
	 * @since 1.0.0
	 */
	public function ajax_activate_license() {

		$this->check_ajax_referer( 'activate-license' );

		$key_rules = array(
			'action'            => 'key',
			'authors_license_key' => 'string',
			'context'           => 'key',
			'nonce'             => 'key',
		);
		$this->set_post_data( $key_rules );

		$api_params = array(
			'edd_action' => 'activate_license',
			'license'    => $this->state_data['authors_license_key'],
			'item_id'    => AUTHORS_ANALYTIFY_PRODUCT_ID, // the name of our product in EDD
			'url'        => home_url(),
		);

		$response = wp_remote_get(
			esc_url_raw( add_query_arg( $api_params, AUTHORS_ANALYTIFY_STORE_URL ) ), array(
				'timeout'   => 15,
				'sslverify' => false,
			)
		);

		// Make sure the response came back okay.
		if ( is_wp_error( $response ) ) {
			return false;
		}

		$license_response = json_decode( wp_remote_retrieve_body( $response ) );
		
		if ( 'valid' === $license_response->license ) {
			$this->set_authors_license_key( $this->state_data['authors_license_key'] );
			$license_response->masked_license = $this->get_formatted_masked_authors_license( $this->state_data['authors_license_key'] );
		} else {
			if ( 'invalid' === $license_response->license ) {
				set_site_transient( 'wpanalytify_authors_license_response', $license_response, $this->transient_timeout );
				$license_response->error = $this->get_authors_license_status_message( $license_response, $this->state_data['context'] );

			}
		}

		// $license_response->license will be either "valid" or "invalid"
		update_option( 'analytify_authors_license_status', $license_response->license );
		update_option( 'analytify_authors_license_key', $this->state_data['authors_license_key'] );

		$result = $this->end_ajax( json_encode( $license_response ) );

		return $result;
	}

}
