<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Analytify_Campaigns_Rest_API' ) ) {

	class Analytify_Campaigns_Rest_API {

		/**
		 * The single instance of the class.
		 *
		 * @var object
		 */
		private static $instance;

		/**
		 * The main Analytify object.
		 *
		 * @var object
		 */
		private $wp_analytify;

		/**
		 * GA version (ga4 or ga3).
		 *
		 * @var string
		 */
		private $ga_mode;

		/**
		 * Selected 'start state'.
		 *
		 * @var string
		 */
		private $start_date;

		/**
		 * Selected 'End state'.
		 *
		 * @var string
		 */
		private $end_date;

		/**
		 * Returns the single instance of the class.
		 *
		 * @return object Class instance
		 */
		public static function get_instance() {
			if ( empty( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Class constructor.
		 */
		private function __construct() {
			add_action( 'rest_api_init', array( $this, 'rest_api_init' ) );
		}

		/**
		 * Register end point.
		 */
		public function rest_api_init() {

			$this->wp_analytify = $GLOBALS['WP_ANALYTIFY'];
			$this->ga_mode      = method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) ? WPANALYTIFY_Utils::get_ga_mode() : 'ga3';

			register_rest_route(
				'wp-analytify/v1',
				'/get_pro_campaigns_report/(?P<request_type>[a-zA-Z0-9-]+)',
				array(
					array(
						'methods'             => WP_REST_Server::READABLE,
						'callback'            => array( $this, 'handle_request' ),
						'permission_callback' => array( $this, 'permission_check' ),
					),
				)
			);
		}

		/**
		 * Checks access permission.
		 * Checks if the user is logged-in and checks of the user role has access.
		 *
		 * @return boolean
		 */
		public function permission_check() {
			$is_access_level = $this->wp_analytify->settings->get_option( 'show_analytics_roles_dashboard', 'wp-analytify-dashboard', array( 'administrator' ) );
			return (bool) $this->wp_analytify->pa_check_roles( $is_access_level );
		}

		/**
		 * Handle the request.
		 *
		 * @param WP_REST_Request $request WP Request object,
		 *
		 * @return array|WP_Error
		 */
		public function handle_request( WP_REST_Request $request ) {

			$this->request_type = $request->get_param( 'request_type' );

			$this->start_date    = $request->get_param( 'sd' );
			$this->end_date      = $request->get_param( 'ed' );
			$this->campaign_name = $request->get_param( 'cn' );

			switch ( $this->request_type ) {
				case 'campaigns':
					return $this->campaigns();
				case 'campaigns-detail':
					return $this->campaigns_detail();
			}

			// If no request type match, Return error.
			return new WP_Error( 'analytify_invalid_endpoint', __( 'Invalid endpoint.', 'wp-analytify' ), array( 'status' => 404 ) );
		}

		/**
		 * Returns all the active campaigns.
		 *
		 * @return string
		 */
		private function campaigns() {

			$api_limit = apply_filters( 'analytify_api_limit_campaigns_stats', 50, 'dashboard' );

			if ( 'ga4' === $this->ga_mode ) {
				$stats = $this->wp_analytify->get_reports(
					'show-campaigns-stats',
					array(
						'sessions',
						'newUsers',
						'bounceRate',
						'averageSessionDuration',
						'screenPageViews',
					),
					array(
						'start' => $this->start_date,
						'end'   => $this->end_date,
					),
					array(
						'sessionCampaignName',
					),
					array(
						'type'  => 'metric',
						'name'  => 'sessions',
						'order' => 'desc',
					),
					array(),
					$api_limit
				);
				include ANALYTIFY_PRO_CAMPAINGS_ROOT_PATH . '/views/campaign-stats.php';
			} else {
				$stats = $this->wp_analytify->pa_get_analytics_dashboard( 'ga:sessions,ga:newUsers,ga:bounces,ga:entrances,ga:avgSessionDuration,ga:pageviews', $this->start_date, $this->end_date, 'ga:campaign', 'ga:campaign', false, false, $api_limit, 'show-campaigns-stats' );
				if ( isset( $stats->totalsForAllResults ) ) {
					include ANALYTIFY_PRO_CAMPAINGS_ROOT_PATH . '/views/campaign-stats-deprecated.php';
				}
			}
		}

		/**
		 * Returns the details for a single campaign.
		 *
		 * @return string
		 */
		private function campaigns_detail() {

			$api_limit = apply_filters( 'analytify_api_limit_campaigns_details_stats', 50, 'dashboard', $this->campaign_name );

			if ( 'ga4' === $this->ga_mode ) {
				$stats = $this->wp_analytify->get_reports(
					'show-campaign-detail-' . $this->campaign_name,
					array(
						'totalUsers',
						'newUsers',
						'sessions',
						'bounceRate',
						'averageSessionDuration',
						'purchaserConversionRate',
						'transactions',
						'purchaseRevenue',
						// 'totalRevenue',
					),
					array(
						'start' => $this->start_date,
						'end'   => $this->end_date,
					),
					array(
						'sessionCampaignName',
						'sessionSourceMedium',
					),
					array(
						'type'  => 'metric',
						'name'  => 'sessions',
						'order' => 'desc',
					),
					array(
						'logic'   => 'AND',
						'filters' => array(
							array(
								'type'       => 'dimension',
								'name'       => 'sessionCampaignName',
								'match_type' => 1,
								'value'      => $this->campaign_name,
							),
						),
					),
					$api_limit
				);

				$html = '<table class="wp_analytify_campaign_paginated" data-product-per-page="10">
				<thead>
					<tr>
						<th>Source/Medium</th>
						<th>Users</th>
						<th>New Users</th>
						<th>Sessions</th>
						<th>Bounce Rate</th>
						<th>Avg. Session Duration</th>
						<th>E-commerce Conversion Rate</th>
						<th>Transactions</th>
						<th>Revenue</th>
					</tr></thead><tbody>';
				foreach ( $stats['rows'] as $key => $data ) {
					$html .= '<tr>
							<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['sessionSourceMedium'] ) . '</td>
							<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['totalUsers'] ) . '</td>
							<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['newUsers'] ) . '</td>
							<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['sessions'] ) . '</td>
							<td>' . number_format( $data['bounceRate'], 2 ) . '%</td>
							<td>' . WPANALYTIFY_Utils::pretty_time( $data['averageSessionDuration'] ) . '</td>
							<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['purchaserConversionRate'] ) . '%</td>
							<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['transactions'] ) . '%</td>
							<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['purchaseRevenue'] ) . '</td>
						</tr>';
				}
			} else {
				$stats = $this->wp_analytify->pa_get_analytics_dashboard( 'ga:users,ga:newUsers,ga:sessions,ga:bounceRate,ga:avgSessionDuration,ga:pageviewsPerSession,ga:transactions,ga:transactionsPerSession,ga:transactionRevenue', $this->start_date, $this->end_date, 'ga:campaign,ga:sourceMedium', 'ga:campaign', 'ga:campaign==' . $this->campaign_name, $api_limit, 'show-campaign-detail-' . $this->campaign_name );

				$html = '<table class="wp_analytify_campaign_paginated" data-product-per-page="10">
				<thead>
					<tr>
						<th>Source/Medium</th>
						<th>Users</th>
						<th>New Users</th>
						<th>Sessions</th>
						<th>Bounce Rate</th>
						<th>Pages/Session</th>
						<th>Avg. Session Duration</th>
						<th>E-commerce Conversion Rate</th>
						<th>Transactions</th>
						<th>Revenue</th>
					</tr></thead><tbody>';
				foreach ( $stats['rows'] as $key => $data ) {
					$html .= '<tr>
							<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['1'] ) . '</td>
							<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['2'] ) . '</td>
							<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['3'] ) . '</td>
							<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['4'] ) . '</td>
							<td>' . number_format( $data['5'], 2 ) . '%</td>
							<td>' . number_format( $data['7'], 2 ) . '</td>
							<td>' . WPANALYTIFY_Utils::pretty_time( $data['6'] ) . '</td>
							<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['8'] ) . '%</td>
							<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['9'] ) . '</td>
							<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['10'] ) . '</td>
						</tr>';
				}
			}

			$html .= '</tbody></table><div class="wp_analytify_campaign_pagination_wraper"></div>';

			wp_send_json(
				array(
					'body' => $html,
				)
			);
		}
	}

	Analytify_Campaigns_Rest_API::get_instance();
}
