<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'ANALYTIFY_PRO_CAMPAIGNS' ) ) {
	/**
	 * Primary class of the add-on.
	 */
	class ANALYTIFY_PRO_CAMPAIGNS extends WP_Analytify_Pro_Base {

		/**
		 * Class instance.
		 *
		 * @var object
		 */
		private static $instance;

		/**
		 * Get the instance of this class.
		 *
		 * @return object
		 */
		public static function get_instance() {
			if ( ! isset( self::$instance ) ) {
				self::$instance = new self();
			}
			return self::$instance;
		}

		/**
		 * Constructor.
		 */
		private function __construct() {
			parent::__construct();
			$this->setup_constants();
			$this->hooks();
		}

		/**
		 * Setup plugin constants
		 *
		 * @access      private
		 * @since       1.0.0
		 * @return      void
		 */
		private function setup_constants() {

			// Setting global values.
			$this->define( 'CAMPAIGNS_ANALYTIFY_PLUGIN_FILE', __FILE__ );
			$this->define( 'CAMPAIGNS_ANALYTIFY_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
			$this->define( 'CAMPAIGNS_ANALYTIFY_PLUGIN_DIR', dirname( __FILE__ ) );
			$this->define( 'CAMPAIGNS_ANALYTIFY_PRODUCT_NAME', 'Google Analytics Campaigns Manager' );
			$this->define( 'CAMPAIGNS_ANALYTIFY_PRODUCT_ID', 6393 );
			$this->define( 'CAMPAIGNS_ANALYTIFY_STORE_URL', 'https://analytify.io' );
		}

		/**
		 * Register hooks.
		 *
		 * @return void
		 */
		private function hooks() {

			add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );

			// License related hooks.
			add_action( 'admin_init', array( $this, 'plugin_updater' ), 0 );
			add_action( 'wp_analytify_campaigns_license_key', array( $this, 'display_license_form' ) );
			add_action( 'wp_ajax_wpanalytify_campaigns_activate_license', array( $this, 'ajax_activate_license' ) );

			// For modifying the meta links.
			add_action( 'plugin_row_meta', array( $this, 'plugin_row_meta' ), 10, 2 );

			add_action( 'analytify_dashboad_dropdown_option', array( $this, 'dashboard_dropdown_option' ) );
			add_action( 'analytify_add_submenu', array( $this, 'add_submenu' ), 30 );

			add_filter( 'analytify_enhanced_general_filter', array( $this, 'analytify_enhanced_general_filter' ) );
			add_filter( 'analytify_enhanced_geographic_filter', array( $this, 'analytify_enhanced_geographic_filter' ) );
			add_filter( 'analytify_enhanced_roi_filter', array( $this, 'analytify_enhanced_roi_filter' ) );
			add_filter( 'analytify_enhanced_coupon_filter', array( $this, 'analytify_enhanced_coupon_filter' ) );
			add_filter( 'analytify_admin_bar_menu', array( $this, 'add_admin_bar_menu' ), 25, 1 );

			add_action( 'page_campaigns', array( $this, 'include_content_campaigns' ) );

		}

		/**
		 * Plugin row meta links.
		 *
		 * @param array  $input Already defined meta links.
		 * @param string $file  Plugin file path and name being processed.
		 * @return array
		 */
		public function plugin_row_meta( $input, $file ) {

			// Exit if its other plugins.
			if ( 'wp-analytify-campaigns/wp-analytify-campaigns.php' !== $file ) {
				return $input;
			}

			// Remove the 'view details' org link.
			unset( $input[2] );

			// Add the plugin site link.
			$input[] = '<a href="https://analytify.io/add-ons/campaigns/">' . esc_html__( 'Visit plugin site', 'wp-analytify-campaings' ) . '</a>';

			return $input;
		}

		/**
		 * Add submenu page for campaigns.
		 *
		 * @since 1.0.0
		 */
		public function add_submenu() {

			add_submenu_page(
				'analytify-dashboard',
				ANALYTIFY_NICK . esc_html__( 'Campaigns', 'wp-analytify-campaings' ),
				esc_html__( 'Campaigns', 'wp-analytify-campaings' ),
				'read',
				'analytify-campaigns',
				array(
					$this,
					'load_main_dashboard',
				),
				20
			);
		}

		/**
		 * Load the main dashboard view.
		 *
		 * @return void
		 */
		public function load_main_dashboard() {
			include_once ANALYTIFY_PRO_CAMPAINGS_ROOT_PATH . '/views/main-dashboard.php';
		}

		/**
		 * Add dropdown option for campaigns dashboard.
		 *
		 * @since 1.0.0
		 */
		public function dashboard_dropdown_option() {
			echo '<li><a href="' . admin_url( 'admin.php?page=analytify-campaigns' ) . '">' . ec_html__( 'Campaigns', 'wp-analytify-campaings' ) . '</a></li>';
		}

		/**
		 * Enqueue admin scripts and styles.
		 *
		 * @param string $page - WP page.
		 * @return void
		 */
		public function admin_scripts( $page ) {
			if ( 'analytify_page_analytify-campaigns' === $page || $page === 'analytify_page_analytify-settings' ) {
				wp_enqueue_script( 'analytify_campaigns_script', plugins_url( 'assets/js/main.js', __DIR__ ), array( 'jquery', 'jquery-ui-tooltip' ), ANALTYIFY_CAMPAIGNS_VERSION );

				// TODO: fix this mess.
				wp_localize_script(
					'analytify_campaigns_script',
					'analytify_campaigns',
					array(
						'resturl'   => wp_json_encode( esc_url_raw( rest_url( 'wp-analytify/v1/get_pro_campaigns_report/campaigns-detail' ) ) ),
						'restnonce' => wp_create_nonce( 'wp_rest' ),
					)
				);
			}
		}

		/**
		 * Change WooCommerce query filter.
		 *
		 * @param string $filter GA filters.
		 *
		 * @since 1.0.4
		 */
		public function analytify_enhanced_general_filter() {
			if ( isset( $_GET['cn'] ) ) {
				$campaign = sanitize_text_field( wp_unslash( $_GET['cn'] ) );
				return 'ga:campaign==' . $campaign;
			}
			return false;
		}

		/**
		 * Change WooCommerce query filter.
		 *
		 * @param string $filter GA filters.
		 *
		 * @since 1.0.4
		 */
		public function analytify_enhanced_geographic_filter( $filter ) {
			if ( isset( $_GET['cn'] ) ) {
				$campaign = sanitize_text_field( wp_unslash( $_GET['cn'] ) );
				$filter  .= ';ga:campaign==' . $campaign;
			}
			return $filter;
		}

		/**
		 * Change WooCommerce query filter.
		 *
		 * @param string $filter GA filters.
		 *
		 * @since 1.0.4
		 */
		public function analytify_enhanced_roi_filter( $filter ) {
			if ( isset( $_GET['cn'] ) ) {
				$campaign = sanitize_text_field( wp_unslash( $_GET['cn'] ) );
				$filter  .= ';ga:campaign==' . $campaign;
			}
			return $filter;
		}

		/**
		 * Change WooCommerce query filter.
		 *
		 * @param string $filter GA filters.
		 *
		 * @since 1.0.4
		 */
		public function analytify_enhanced_coupon_filter( $filter ) {
			if ( isset( $_GET['cn'] ) ) {
				$campaign = sanitize_text_field( wp_unslash( $_GET['cn'] ) );
				$filter  .= ';ga:campaign==' . $campaign;
			}
			return $filter;
		}

		/**
		 * Add option in admin bar.
		 *
		 * @param array $menus Menu array.
		 *
		 * @since 1.0.5
		 */
		public function add_admin_bar_menu( $menus ) {

			$menus['analytify-campaigns'] = __( 'Campaigns', 'wp-analytify-campaings' );
			return $menus;
		}

		/**
		 * Define constant if not already set
		 *
		 * @param string      $name  Constant name.
		 * @param string|bool $value Constant value.
		 */
		private function define( $name, $value ) {
			if ( ! defined( $name ) ) {
				define( $name, $value );
			}
		}

		/**
		 * Update your plugin from our own server.
		 *
		 * @since 1.0.0
		 */
		public function plugin_updater() {

			// Retrieve our license key from the DB.
			$license_key = trim( get_option( 'analytify_campaigns_license_key' ) );

			// Setup the updater.
			if ( class_exists( 'ANALYTIFY_SL_Plugin_Updater' ) ) {
				$edd_updater = new ANALYTIFY_SL_Plugin_Updater(
					CAMPAIGNS_ANALYTIFY_STORE_URL,
					ANALYTIFY_CAMPAIGNS_UPGRADE_PATH,
					array(
						// Current version number.
						'version' => ANALTYIFY_CAMPAIGNS_VERSION,
						// License key (used get_option above to retrieve from DB).
						'license' => $license_key,
						// Name of this plugin.
						'item_id' => CAMPAIGNS_ANALYTIFY_PRODUCT_ID,
						// author of this plugin.
						'author'  => 'Muhammad Adnan',
					)
				);
			}
		}

		/**
		 * Output plugin license activation key fields/inputs.
		 *
		 * @return void
		 */
		public function display_license_form() {

			$license_campaigns = get_option( 'analytify_campaigns_license_key' );
			$status_campaigns  = get_option( 'analytify_campaigns_license_status' );
			?>
			<tr valign="top">
				<th scope="row" valign="top"><?php esc_html_e( 'Campaigns Tracking:', 'wp-analytify-campaings' ); ?></th>

				<?php
				if ( 'valid' === $status_campaigns ) {
					?>
					<td class="campaigns-license-row">
						<?php echo $this->get_formatted_masked_campaigns_license( $license_campaigns ); ?>
					</td>
					<?php
				} else {
					?>
					<td class="campaigns-license-row">
						<input id="analytify_campaigns_license_key" name="analytify_campaigns_license_key" type="text" class="regular-text" value="<?php echo esc_attr( $license_campaigns ); ?>" />

						<input type="submit" class="button-secondary" id="analytify_campaigns_license_activate" name="analytify_license_activate" value="<?php esc_html_e( 'Activate License', 'wp-analytify-campaings' ); ?>" />

						<?php
						if ( $status_campaigns ) {
							?>
							<br />
							<p id="campaigns-license-status">
								<?php echo esc_html( $status_campaigns ); ?>
							</p>
							<?php
						}
						?>

					</td>
					<?php
				}
				?>
			</tr>
			<?php
		}

		/**
		 *
		 * Send Ajax on Active license.
		 *
		 * @since 1.0.0
		 */
		public function ajax_activate_license() {

			$this->check_ajax_referer( 'activate-license' );

			$key_rules = array(
				'action'                => 'key',
				'campaigns_license_key' => 'string',
				'context'               => 'key',
				'nonce'                 => 'key',
			);
			$this->set_post_data( $key_rules );

			$api_params = array(
				'edd_action' => 'activate_license',
				'license'    => $this->state_data['campaigns_license_key'],
				'item_name'  => urlencode( CAMPAIGNS_ANALYTIFY_PRODUCT_NAME ),
				'url'        => home_url(),
			);

			$response = wp_remote_get(
				esc_url_raw( add_query_arg( $api_params, CAMPAIGNS_ANALYTIFY_STORE_URL ) ),
				array(
					'timeout'   => 30,
					'sslverify' => false,
				)
			);

			// Make sure the response comes back okay.
			if ( is_wp_error( $response ) ) {
				return false;
			}

			$license_response = json_decode( wp_remote_retrieve_body( $response ) );

			if ( 'valid' === $license_response->license ) {
				$this->set_campaigns_license_key( $this->state_data['campaigns_license_key'] );
				$license_response->masked_license = $this->get_formatted_masked_campaigns_license( $this->state_data['campaigns_license_key'] );
			} else {
				if ( 'invalid' === $license_response->license ) {

					set_site_transient( 'wpanalytify_campaigns_license_response', $license_response, $this->transient_timeout );
					$license_response->error = $this->get_campaigns_license_status_message( $license_response, $this->state_data['context'] );

				}
			}

			// $license_response->license will be either "valid" or "invalid"
			update_option( 'analytify_campaigns_license_status', $license_response->license );
			update_option( 'analytify_campaigns_license_key', $this->state_data['campaigns_license_key'] );

			$result = $this->end_ajax( wp_json_encode( $license_response ) );

			return $result;
		}
	}

	ANALYTIFY_PRO_CAMPAIGNS::get_instance();
}
