<?php
class WP_Analytify_Email extends WP_Analytify_Pro_Base {

	/**
	 * Constructor.
	 */
	public function __construct() {
		parent::__construct();
		$this->setup_constants();
		$this->hooks();
	}

	/**
	 * Register all the hooks.
	 *
	 * @return void
	 */
	private function hooks() {
		add_action( 'admin_init', array( $this, 'save_version' ), 50 );
		add_action( 'admin_init', array( $this, 'update_routine' ), 10 );
		add_action( 'admin_init', array( $this, 'backward_compatibility' ), 10 );

		add_action( 'admin_init', array( $this, 'plugin_updater' ), 0 );
		add_action( 'wp_analytify_email_license_key', array( $this, 'display_license_form' ) );

		add_filter( 'wp_analytify_pro_setting_fields', array( $this, 'analytify_email_setting_fields' ), 21, 1 );
		add_action( 'wp_ajax_wpanalytifyemail_activate_license', array( $this, 'ajax_activate_license' ) );

		/**
		 * This hooks is called in the Core version.
		 */
		add_filter( 'wp_analytify_email_on_cron_time', array( $this, 'callback_wp_analytify_email_on_cron_time' ), 10, 3 );
		add_filter( 'wp_analytify_single_email', array( $this, 'callback_wp_analytify_single_email' ), 10, 3 );
	}

	/**
	 * Setup plugin constants
	 *
	 * @access      private
	 * @since       1.0.0
	 * @return      void
	 */
	private function setup_constants() {
		// Setting Global Values.
		$this->define( 'EMAIL_ANALYTIFY_PLUGIN_FILE', __FILE__ );
		$this->define( 'EMAIL_ANALYTIFY_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
		$this->define( 'EMAIL_ANALYTIFY_PLUGIN_DIR', dirname( __FILE__ ) );
		$this->define( 'EMAIL_ANALYTIFY_PRODUCT_NAME', 'Google Analytics Email Notifications' );
		$this->define( 'EMAIL_ANALYTIFY_PRODUCT_ID', 6580 );
		$this->define( 'EMAIL_ANALYTIFY_STORE_URL', 'https://analytify.io' );
	}

	/**
	 * Define constant if not already set.
	 *
	 * @param  string      $name
	 * @param  string|bool $value
	 */
	private function define( $name, $value ) {
		if ( ! defined( $name ) ) {
			define( $name, $value );
		}
	}

	public function analytify_email_notics() {
		?>
		<div class="notice notice-success is-dismissible">
			<p><?php _e( 'Analytify detailed report sent!', 'wp-analytify-email' ); ?></p>
		</div>
		<?php
	}

	public function custom_phpmailer_init( $PHPMailer ) {
		$PHPMailer->IsSMTP();
		$PHPMailer->SMTPAuth   = true;
		$PHPMailer->SMTPSecure = 'ssl';
		$PHPMailer->Host       = 'smtp.gmail.com';
		$PHPMailer->Port       = 465;
		$PHPMailer->Username   = 'test@gmail.com';
		$PHPMailer->Password   = '';
	}

	public function analytify_email_setting_fields( $old_fields ) {

		$stats_option = array(
			'show-overall-general' => analytify__( 'General Stats', 'wp-analytify' ),
			'show-top-pages'       => __( 'Top Pages Stats', 'wp-analytify-email' ),
			'show-geographics'     => analytify__( 'Geographic Stats', 'wp-analytify' ),
			'show-system'          => analytify__( 'System Stats', 'wp-analytify' ),
			'show-keywords'        => analytify__( 'Keywords Stats', 'wp-analytify' ),
			'show-social'          => analytify__( 'Social Media Stats', 'wp-analytify' ),
			'show-whats-happen'    => __( 'Whats Happening Stats', 'wp-analytify-email' ),
			'show-referrer'        => analytify__( 'Referrers Stats', 'wp-analytify' ),
		);

		// check if WooCommerce or EDD installed.
		if ( class_exists( 'WP_Analytify_WooCommerce' ) || class_exists( 'WP_Analytify_Edd' ) ) {
			$stats_option['show-e-commerce-stats'] = __( 'E-Commerce Report', 'wp-analytify-email' );
		}

		$free_email_fields = $old_fields['wp-analytify-email'];

		// New addon fields.
		$addon_fields = array(
			array(
				'name'    => 'analytif_email_cron_time',
				'label'   => __( 'Set Schedule', 'wp-analytify-email' ),
				'desc'    => __( 'Select days to schedule weekly or/and monthly emails.', 'wp-analytify-email' ),
				'type'    => 'multi_select',
				'default' => 'weekly',
				'options' => array(
					'main'  => array(
						'week'  => 'Weekly',
						'month' => 'Monthly',
					),
					'value' => array(
						'week'  => array(
							'false'     => __( 'Select Day', 'wp-analytify-email' ),
							'Sunday'    => analytify__( 'Sunday' ),
							'Monday'    => analytify__( 'Monday' ),
							'Tuesday'   => analytify__( 'Tuesday' ),
							'Wednesday' => analytify__( 'Wednesday' ),
							'Thursday'  => analytify__( 'Thursday' ),
							'Friday'    => analytify__( 'Friday' ),
							'Saturday'  => analytify__( 'Saturday' ),
						),
						'month' => array(
							'false'    => __( 'Select Date', 'wp-analytify-email' ),
							'1'        => __( '1', 'wp-analytify-email' ),
							'2'        => __( '2', 'wp-analytify-email' ),
							'3'        => __( '3', 'wp-analytify-email' ),
							'4'        => __( '4', 'wp-analytify-email' ),
							'5'        => __( '5', 'wp-analytify-email' ),
							'6'        => __( '6', 'wp-analytify-email' ),
							'7'        => __( '7', 'wp-analytify-email' ),
							'8'        => __( '8', 'wp-analytify-email' ),
							'9'        => __( '9', 'wp-analytify-email' ),
							'10'       => __( '10', 'wp-analytify-email' ),
							'11'       => __( '11', 'wp-analytify-email' ),
							'12'       => __( '12', 'wp-analytify-email' ),
							'13'       => __( '13', 'wp-analytify-email' ),
							'14'       => __( '14', 'wp-analytify-email' ),
							'15'       => __( '15', 'wp-analytify-email' ),
							'16'       => __( '16', 'wp-analytify-email' ),
							'17'       => __( '17', 'wp-analytify-email' ),
							'18'       => __( '18', 'wp-analytify-email' ),
							'19'       => __( '19', 'wp-analytify-email' ),
							'20'       => __( '20', 'wp-analytify-email' ),
							'21'       => __( '21', 'wp-analytify-email' ),
							'22'       => __( '22', 'wp-analytify-email' ),
							'23'       => __( '23', 'wp-analytify-email' ),
							'24'       => __( '24', 'wp-analytify-email' ),
							'25'       => __( '25', 'wp-analytify-email' ),
							'26'       => __( '26', 'wp-analytify-email' ),
							'27'       => __( '27', 'wp-analytify-email' ),
							'28'       => __( '28', 'wp-analytify-email' ),
							'last_day' => __( 'Last Day', 'wp-analytify-email' ),
						),
					),
				),
			),
			array(
				'name'              => 'analytify_email_subject',
				'label'             => __( 'Subject', 'wp-analytify-email' ),
				'desc'              => __( 'Add Email Subject', 'wp-analytify-email' ),
				'type'              => 'text',
				'default'           => '',
				'sanitize_callback' => 'sanitize_text_field',
			),
			array(
				'name'    => 'analytify_email_stats',
				'label'   => __( 'Select Email Stats', 'wp-analytify-email' ),
				'desc'    => __( 'Select which Stats panels you want to display in Email. ', 'wp-analytify-email' ),
				'type'    => 'chosen',
				'default' => array(),
				'options' => $stats_option,
			),
			array(
				'name'              => 'analytify_email_logo',
				'label'             => __( 'Your Agency Logo', 'wp-analytify-email' ),
				'desc'              => __( 'Add your Agency logo in Email Template.<br />Preferred size 150*150 pixel', 'wp-analytify-email' ),
				'type'              => 'image',
				'default'           => '',
				'sanitize_callback' => '',
			),
			array(
				'name'              => 'analytiy_mail_text',
				'label'             => __( 'Personal Note', 'wp-analytify-email' ),
				'desc'              => __( 'The personal message will be added in Email Report. Leave it empty to disable it.', 'wp-analytify-email' ),
				'type'              => 'wysiwyg',
				'default'           => '',
				'options'           => array(
					'tinymce'       => false,
					'media_buttons' => false,
				),
				'sanitize_callback' => 'wp_kses_post',
			),
			array(
				'name'              => 'analytiy_mail_text_additional',
				'label'             => __( 'Additional Note', 'wp-analytify-email' ),
				'desc'              => __( 'An additional note area at the bottom of the email. Leave it empty to disable it.', 'wp-analytify-email' ),
				'type'              => 'wysiwyg',
				'options'           => array(
					'tinymce'       => false,
					'media_buttons' => false,
				),
				'default'           => __( 'We hope your weekly Analytics summaries encourage the efforts you are putting in your business. But if you no longer wish to receive Analytify Insights emails, you may change your preferences from plugin settings.', 'wp-analytify-email' ),
				'sanitize_callback' => 'wp_kses_post',
			),
			array(
				'name'  => 'analytify_go_to_dashboard',
				'label' => __( 'Hide \'Go To Dashboard\' Button:', 'wp-analytify-email' ),
				'desc'  => __( 'Check this if you want to hide the Go To Dashboard button', 'wp-analytify-email' ),
				'type'  => 'checkbox',
			),
		);

		// Add new addon fields.
		$pro_fields = array_merge( $free_email_fields, $addon_fields );

		// Add sender name option at second position.
		array_splice(
			$pro_fields,
			1,
			0,
			array(
				array(
					'name'              => 'analytiy_from_name',
					'label'             => __( 'From Name', 'wp-analytify-email' ),
					'desc'              => __( 'Sender Name.', 'wp-analytify-email' ),
					'type'              => 'text',
					'default'           => '',
					'sanitize_callback' => 'sanitize_text_field',
				),
			)
		);

		$old_fields['wp-analytify-email'] = $pro_fields;

		return $old_fields;
	}

	public function save_version() {

		if ( ANALTYIFY_EMAIL_VERSION != get_option( 'ANALTYIFY_EMAIL_VERSION' ) ) {
			update_option( 'ANALTYIFY_EMAIL_VERSION', ANALTYIFY_EMAIL_VERSION );
		}
	}

	public function update_routine() {

		$old_version  = get_option( 'ANALTYIFY_EMAIL_VERSION' );
		$email_option = get_option( 'wp-analytify-email' );

		// change option structure.
		if ( ! $old_version && ! isset( $analytif_email_cron_time['analytif_email_cron_time']['value'] ) ) {
			if ( $email_option['analytif_email_cron_time'] == 'weekly' || $email_option['analytif_email_cron_time'] == 'biweekly' ) {
				$email_option['analytif_email_cron_time'] = array(
					'week'  => 'Monday',
					'month' => 'false',
				);
			} elseif ( $email_option['analytif_email_cron_time'] == '4weeks' ) {
				$email_option['analytif_email_cron_time'] = array(
					'week'  => 'false',
					'month' => '1',
				);
			}

			update_option( 'wp-analytify-email', $email_option );
			// clear old cron time.
			wp_clear_scheduled_hook( 'analytify_email_cron_function' );
		}
	}

	/**
	 * Add pro reports in email message body.
	 *
	 * @return $message
	 */
	public function callback_wp_analytify_email_on_cron_time( $message, $selected_sections, $dates ) {

		if ( ! is_array( $selected_sections ) || empty( $selected_sections ) ) {
			return $message;
		}

		$wp_analytify = $GLOBALS['WP_ANALYTIFY'];

		// Ecommerce.
		if ( in_array( 'show-e-commerce-stats', $selected_sections, true ) ) {

			// WooCommerce.
			if ( class_exists( 'WP_Analytify_WooCommerce' ) ) {
				if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {

					$general_stats = $wp_analytify->get_reports(
						'analytify-email-woo-general-stats',
						array(
							'transactions',
							'totalRevenue',
							'averagePurchaseRevenue',
							'ecommercePurchases',
							'addToCarts',
							'checkouts',
							'itemListClicks',
							'itemViews',
						),
						array(
							'start' => $dates['start_date'],
							'end'   => $dates['end_date'],
						),
						array(),
						array(),
						array(),
						0
					);

					$remove_from_cart = $wp_analytify->get_reports(
						'analytify-email-woo-remove-from-cart',
						array(
							'eventCount',
						),
						array(
							'start' => $dates['start_date'],
							'end'   => $dates['end_date'],
						),
						array(
							'eventName',
						),
						array(),
						array(
							'logic'   => 'AND',
							'filters' => array(
								array(
									'type'       => 'dimension',
									'name'       => 'eventName',
									'match_type' => 1,
									'value'      => 'remove_from_cart',
								),
							),
						),
						0
					);

					if ( isset( $remove_from_cart['aggregations']['eventCount'] ) ) {
						$general_stats['remove_from_cart'] = $remove_from_cart['aggregations']['eventCount'];
					}
					if ( isset( $general_stats['aggregations'] ) ) {
						include ANALYTIFY_EMAIL_ROOT_PATH . '/views/woo-general-stats.php';
						$message .= pa_email_include_woo_general( $this->wp_analytify, $general_stats );
					}
				} else {
					$stats = $wp_analytify->pa_get_analytics_dashboard( 'ga:transactions,ga:transactionRevenue,ga:revenuePerTransaction,ga:uniquePurchases,ga:transactionsPerSession,ga:productAddsToCart,ga:productRemovesFromCart,ga:productCheckouts,ga:productListClicks,ga:productDetailViews', $dates['start_date'], $dates['end_date'], false, false, false, false, 'analytify-email-woo-general-stats' );

					if ( ! function_exists( 'pa_email_include_woo_general' ) ) {
						include ANALYTIFY_EMAIL_ROOT_PATH . '/views/woo-general-stats-deprecated.php';
						$message .= pa_email_include_woo_general( $wp_analytify, $stats );
					}
				}
			}

			// EDD.
			if ( class_exists( 'WP_Analytify_Edd' ) ) {
				if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {

					$stats = $wp_analytify->get_reports(
						'edd-shopping-analysis-stats',
						array(
							'transactions',
							'totalRevenue',
							'averagePurchaseRevenue',
							// 'ga:refundAmount, TODO: missing_ga4
							// 'ga:transactionsPerSession, TODO: missing_ga4
							'addToCarts',
							// 'ga:productRemovesFromCart, TODO: missing_ga4
							'checkouts',
							'itemViews',
							// 'ga:totalRefunds, TODO: missing_ga4
						),
						array(
							'start' => $dates['start_date'],
							'end'   => $dates['end_date'],
						),
						array(),
						array(),
						array(),
						0
					);

					if ( ! function_exists( 'pa_email_include_edd_general' ) ) {
						include ANALYTIFY_EMAIL_ROOT_PATH . '/views/edd-general-stats.php';
						$message .= pa_email_include_edd_general( $wp_analytify, $stats );
					}
				} else {
					$stats = $wp_analytify->pa_get_analytics_dashboard( 'ga:transactions,ga:transactionRevenue,ga:revenuePerTransaction,ga:refundAmount,ga:transactionsPerSession,ga:productAddsToCart,ga:productRemovesFromCart,ga:productCheckouts,ga:productDetailViews,ga:totalRefunds', $dates['start_date'], $dates['end_date'], false, false, false, false, false, 'analytify-email-edd-general-stats' );

					if ( ! function_exists( 'pa_email_include_edd_general' ) ) {
						include ANALYTIFY_EMAIL_ROOT_PATH . '/views/edd-general-stats-deprecated.php';
						$message .= pa_email_include_edd_general( $wp_analytify, $stats );
					}
				}
			}
		}

		// Top Pages Stats.
		if ( in_array( 'show-top-pages', $selected_sections, true ) ) {
			if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {

				$stats = $wp_analytify->get_reports(
					'analytify-email-top-pages-stats',
					array(
						'screenPageViews',
						'userEngagementDuration',
						'bounceRate',
					),
					array(
						'start' => $dates['start_date'],
						'end'   => $dates['end_date'],
					),
					array(
						'pageTitle',
						'pagePath',
					),
					array(
						'type'  => 'metric',
						'name'  => 'screenPageViews',
						'order' => 'desc',
					),
					array(
						'logic'   => 'AND',
						'filters' => array(
							array(
								'type'           => 'dimension',
								'name'           => 'pageTitle',
								'match_type'     => 4,
								'value'          => '(not set)',
								'not_expression' => true,
							),
							array(
								'type'           => 'dimension',
								'name'           => 'pagePath',
								'match_type'     => 4,
								'value'          => '(not set)',
								'not_expression' => true,
							),
						),
					),
					5,
				);

				if ( ! function_exists( 'pa_include_top_pages_stats' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/top-pages-stats.php';
					$message .= pa_include_top_pages_stats( $wp_analytify, $stats );
				}
			} else {
				$stats = $wp_analytify->pa_get_analytics_dashboard( 'ga:pageviews', $dates['start_date'], $dates['end_date'], 'ga:PageTitle', '-ga:pageviews', false, 5, 'analytify-email-top-pages-stats' );

				if ( ! function_exists( 'pa_include_top_pages_stats' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/top-pages-stats-deprecated.php';
					$message .= pa_include_top_pages_stats( $wp_analytify, $stats );
				}
			}
		}

		// Geographic Stats
		if ( in_array( 'show-geographics', $selected_sections, true ) ) {
			if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {
				$countries_stats = $wp_analytify->get_reports(
					'analytify-email-country-stats',
					array(
						'sessions',
					),
					array(
						'start' => $dates['start_date'],
						'end'   => $dates['end_date'],
					),
					array(
						'country',
					),
					array(
						'type'  => 'metric',
						'name'  => 'sessions',
						'order' => 'desc',
					),
					array(
						'logic'   => 'AND',
						'filters' => array(
							array(
								'type'           => 'dimension',
								'name'           => 'country',
								'match_type'     => 4,
								'value'          => '(not set)',
								'not_expression' => true,
							),
						),
					),
					10
				);

				$cities_stats = $wp_analytify->get_reports(
					'analytify-email-city-stats',
					array(
						'sessions',
					),
					array(
						'start' => $dates['start_date'],
						'end'   => $dates['end_date'],
					),
					array(
						'city',
						'country',
					),
					array(
						'type'  => 'metric',
						'name'  => 'sessions',
						'order' => 'desc',
					),
					array(
						'logic'   => 'AND',
						'filters' => array(
							array(
								'type'           => 'dimension',
								'name'           => 'country',
								'match_type'     => 4,
								'value'          => '(not set)',
								'not_expression' => true,
							),
							array(
								'type'           => 'dimension',
								'name'           => 'city',
								'match_type'     => 4,
								'value'          => '(not set)',
								'not_expression' => true,
							),
						),
					),
					10
				);

				if ( ! function_exists( 'pa_geopgraphic_lisiting' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/geographic.php';
				}

				$message .= pa_geopgraphic_lisiting( $wp_analytify, $countries_stats, $cities_stats );
			} else {
				$countries = $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:country', '-ga:sessions', 'ga:country!=(not set)', 5, 'analytify-email-country-stats' );
				$cities    = $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:country,ga:city', '-ga:sessions', 'ga:city!=(not set)', 5, 'analytify-email-city-stats' );

				if ( ! function_exists( 'pa_geopgraphic_lisiting' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/geographic-deprecated.php';
				}

				$message .= pa_geopgraphic_lisiting( $wp_analytify, $countries, $cities );
			}
		}

		// System stats
		if ( in_array( 'show-system', $selected_sections, true ) ) {

			if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {
				$browser_stats = $wp_analytify->get_reports(
					'show-default-browser-dashboard',
					array(
						'sessions',
					),
					array(
						'start' => $dates['start_date'],
						'end'   => $dates['end_date'],
					),
					array(
						'browser',
						'operatingSystem',
					),
					array(
						'type'  => 'metric',
						'name'  => 'sessions',
						'order' => 'desc',
					),
					array(
						'logic'   => 'AND',
						'filters' => array(
							array(
								'type'           => 'dimension',
								'name'           => 'operatingSystem',
								'match_type'     => 4,
								'value'          => '(not set)',
								'not_expression' => true,
							),
						),
					),
					5
				);

				$os_stats = $wp_analytify->get_reports(
					'show-default-os-dashboard',
					array(
						'sessions',
					),
					array(
						'start' => $dates['start_date'],
						'end'   => $dates['end_date'],
					),
					array(
						'operatingSystem',
						'operatingSystemVersion',
					),
					array(
						'type'  => 'metric',
						'name'  => 'sessions',
						'order' => 'desc',
					),
					array(
						'logic'   => 'AND',
						'filters' => array(
							array(
								'type'           => 'dimension',
								'name'           => 'operatingSystemVersion',
								'match_type'     => 4,
								'value'          => '(not set)',
								'not_expression' => true,
							),
						),
					),
					5
				);

				$mobile_stats = $wp_analytify->get_reports(
					'show-default-mobile-dashboard',
					array(
						'sessions',
					),
					array(
						'start' => $dates['start_date'],
						'end'   => $dates['end_date'],
					),
					array(
						'mobileDeviceBranding',
						'mobileDeviceModel',
					),
					array(
						'type'  => 'metric',
						'name'  => 'sessions',
						'order' => 'desc',
					),
					array(
						'logic'   => 'AND',
						'filters' => array(
							array(
								'type'           => 'dimension',
								'name'           => 'deviceCategory',
								'match_type'     => 4,
								'value'          => 'desktop',
								'not_expression' => true,
							),
							array(
								'type'           => 'dimension',
								'name'           => 'mobileDeviceModel',
								'match_type'     => 4,
								'value'          => '(not set)',
								'not_expression' => true,
							),
						),
					),
					5
				);

				if ( ! function_exists( 'pa_system_stats' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/system-stats.php';
					$message .= pa_system_stats( $wp_analytify, $browser_stats, $os_stats, $mobile_stats );
				}
			} else {
				$browers           = $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:browser', '-ga:sessions', false, 5, 'analytify-email-browser-stats' );
				$operating_systems = $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:operatingSystem', '-ga:sessions', 'ga:operatingSystem!=(not set)', 5, 'analytify-email-system-stats' );
				$mobile_phones     = $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:mobileDeviceBranding,ga:mobileDeviceModel', '-ga:sessions', 'ga:mobileDeviceModel!=(not set);ga:mobileDeviceBranding!=(not set)', 5, 'analytify-email-mobile-stats' );

				if ( ! function_exists( 'pa_system_stats' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/system-stats-deprecated.php';
					$message .= pa_system_stats( $wp_analytify, $browers, $operating_systems, $mobile_phones );
				}
			}
		}

		// Keywords and/or refferals
		if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {

			$keyword_stats_raw = $wp_analytify->get_search_console_stats(
				'analytify-email-keyword-stats',
				array(
					'start' => $dates['start_date'],
					'end'   => $dates['end_date'],
				),
				5
			);

			$keywords = false;
			if ( isset( $keyword_stats_raw['response']['rows'] ) && $keyword_stats_raw['response']['rows'] > 0 ) {
				$keywords = $keyword_stats_raw['response']['rows'];
			}

			$referer_stats_raw = $wp_analytify->get_reports(
				'analytify-email-refferal-stats',
				array(
					'sessions',
				),
				array(
					'start' => $dates['start_date'],
					'end'   => $dates['end_date'],
				),
				array(
					'sessionSource',
					'sessionMedium',
				),
				array(
					'type'  => 'metric',
					'name'  => 'sessions',
					'order' => 'desc',
				),
				array(),
				5
			);

			$refferals = false;

			if ( isset( $refferals ) ) {
				$refferals = $referer_stats_raw;
			}

			if ( ! function_exists( 'pa_keyword_referral' ) ) {
				include ANALYTIFY_EMAIL_ROOT_PATH . '/views/keyword-referral.php';
				$message .= pa_keyword_referral( $wp_analytify, $keywords, $refferals );
			}
		} else {
			// check if the 'keywords' are set
			$keywords = ( in_array( 'show-keywords', $selected_sections, true ) ) ? $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:keyword', '-ga:sessions', false, 5, 'analytify-email-keyword-stats' ) : false;

			// check if the 'referrer' are set
			$refferals = ( in_array( 'show-referrer', $selected_sections, true ) ) ? $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:source,ga:medium', '-ga:sessions', false, 5, 'analytify-email-refferal-stats' ) : false;

			// add stats to the email if either on is set
			if ( $keywords or $refferals ) {
				if ( ! function_exists( 'pa_keyword_referral' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/keyword-referral-deprecated.php';
					$message .= pa_keyword_referral( $wp_analytify, $keywords, $refferals );
				}
			}
		}

		// Social Media
		if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {
			// TODO: missing_ga4
		} else {
			if ( in_array( 'show-social', $selected_sections, true ) ) {
				$socials = $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:socialNetwork', '-ga:sessions', 'ga:socialNetwork!=(not set)', 5, 'analytify-email-social-media-stats' );

				if ( ! function_exists( 'social_stats' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/social-media.php';
				}
				$message .= social_stats( $wp_analytify, $socials );
			}
		}

		// Whats Happen
		if ( in_array( 'show-whats-happen', $selected_sections, true ) ) {
			if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {
				// jawad changes this var name from stats to whats_happens finally
				$whats_happens = $wp_analytify->get_reports(
					'analytify-email-what-happen-stats',
					array(
						'engagedSessions',
						'engagementRate',
						'userEngagementDuration',
					),
					array(
						'start' => $dates['start_date'],
						'end'   => $dates['end_date'],
					),
					array(
						'landingPage',
						'pageTitle',
					),
					array(
						'type'  => 'metric',
						'name'  => 'engagedSessions',
						'order' => 'desc',
					),
					array(),
					5
				);
				if ( ! function_exists( 'whats_happen' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/whats-happen.php';
					$message .= whats_happen( $wp_analytify, $whats_happens ); 
				}
			} else {
				$whats_happens = $wp_analytify->pa_get_analytics_dashboard( 'ga:entrances,ga:entranceRate,ga:exits,ga:exitRate', $dates['start_date'], $dates['end_date'], 'ga:pagePath,ga:pageTitle', '-ga:entrances', false, 5, 'analytify-email-what-happen-stats' );
				if ( ! function_exists( 'whats_happen' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/whats-happen-deprecated.php';
					$message .= whats_happen( $wp_analytify, $whats_happens );
				}
			}
		}

		if ( '' !== $wp_analytify->settings->get_option( 'analytiy_mail_text', 'wp-analytify-email' ) ) {
			$message .= '<tr>
			<td style="padding: 40px 20px; text-align:center;padding-bottom: 20px;font: normal 13px \'Roboto\', Arial, Helvetica, sans-serif;"><font color="#444444">' . $wp_analytify->settings->get_option( 'analytiy_mail_text', 'wp-analytify-email' ) . '</font></td>
			</tr>';
		}

		// Should we add the email note.
		$show_email_note = apply_filters( 'wp_analytify_show_email_note', true );
		$emil_note_text  = $wp_analytify->settings->get_option( 'analytiy_mail_text_additional', 'wp-analytify-email' );

		if ( '' !== $emil_note_text && $show_email_note ) {
			// Filter the email additional note text.
			$text_email_note = apply_filters( 'wp_analytify_email_note_text', $emil_note_text );

			if ( $show_email_note ) {
				$note_title        = apply_filters( 'wp_analytify_email_note_title', 'Note:' );
				$note_title_string = ! empty( $note_title ) ? '<td valign="top" style="width:100px; font: bold 14px \'Roboto\', Arial, Helvetica, sans-serif; padding-right:4px;"><font  color="#444444">' . $note_title . ' </font></td>' : '';

				$message .= '
				<tr>
					<td style="padding: 20px 20px;">
						<table cellpadding="0" cellspacing="0" border="0" width="100%">
							<tr>
								' . $note_title_string . '
								<td style="font: normal 14px \'Roboto\', Arial, Helvetica, sans-serif;"><font color="#848484">' . $text_email_note . '</font><br><br></td>
							</tr>
						</table>
					</td>
				</tr>';
			}
		}

		if ( 'on' !== $wp_analytify->settings->get_option( 'analytify_go_to_dashboard', 'wp-analytify-email' ) ) {
			$dashboard_link = apply_filters( 'analytify_email_dashboard_url', get_admin_url( null, 'admin.php?page=analytify-dashboard' ) );
			$message       .= '<tr>
				<td style="padding-top: 40px; text-align:center;padding-bottom: 40px;"><a href="' . esc_url( $dashboard_link ) . '"><img src="' . ANALYTIFY_IMAGES_PATH . 'analytify_go_to_dashboard.png" alt="go to dashboard"  /></a></td>
			</tr>';
		}

		return $message;
	}

	/**
	 * Add pro reports in single post email message body.
	 *
	 * @return $message
	 */
	public function callback_wp_analytify_single_email( $message, $selected_sections, $dates ) {

		if ( ! is_array( $selected_sections ) || empty( $selected_sections ) ) {
			return $message;
		}

		$post_id      = sanitize_text_field( wp_unslash( $_POST['post_id'] ) );
		$post_url     = get_the_permalink( $post_id );
		$u_post       = parse_url( urldecode( $post_url ) );
		$filter       = $u_post['path'];
		$filter       = apply_filters( 'analytify_page_path_filter', $filter, $u_post );
		// Url have query string incase of WPML.
		if ( isset( $u_post['query'] )  ) {
			$filter .= '?' . $u_post['query'];
		}

		$wp_analytify = $GLOBALS['WP_ANALYTIFY'];
		$ga_mode      = method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) ? WPANALYTIFY_Utils::get_ga_mode() : 'ga3';

		// Geographic Stats.
		if ( in_array( 'show-geographics', $selected_sections, true ) ) {
			if ( 'ga4' === $ga_mode ) {
				$countries_stats = $wp_analytify->get_reports(
					'analytify-email-country-stats',
					array(
						'sessions',
					),
					array(
						'start' => $dates['start_date'],
						'end'   => $dates['end_date'],
					),
					array(
						'country',
						'pagePath',
					),
					array(
						'type'  => 'metric',
						'name'  => 'sessions',
						'order' => 'desc',
					),
					array(
						'logic'   => 'AND',
						'filters' => array(
							array(
								'type'           => 'dimension',
								'name'           => 'country',
								'match_type'     => 4,
								'value'          => '(not set)',
								'not_expression' => true,
							),
							array(
								'type'       => 'dimension',
								'name'       => 'pagePath',
								'match_type' => 1,
								'value'      => $filter,
							),
						),
					)
				);

				$cities_stats = $wp_analytify->get_reports(
					'analytify-email-city-stats',
					array(
						'sessions',
					),
					array(
						'start' => $dates['start_date'],
						'end'   => $dates['end_date'],
					),
					array(
						'city',
						'country',
						'pagePath',
					),
					array(
						'type'  => 'metric',
						'name'  => 'sessions',
						'order' => 'desc',
					),
					array(
						'logic'   => 'AND',
						'filters' => array(
							array(
								'type'           => 'dimension',
								'name'           => 'country',
								'match_type'     => 4,
								'value'          => '(not set)',
								'not_expression' => true,
							),
							array(
								'type'           => 'dimension',
								'name'           => 'city',
								'match_type'     => 4,
								'value'          => '(not set)',
								'not_expression' => true,
							),
							array(
								'type'       => 'dimension',
								'name'       => 'pagePath',
								'match_type' => 1,
								'value'      => $filter,
							),
						),
					)
				);
				if ( ! function_exists( 'pa_geopgraphic_lisiting' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/geographic.php';
				}
				$message .= pa_geopgraphic_lisiting( $wp_analytify, $countries_stats, $cities_stats );
			} else {
				$countries = $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:country', '-ga:sessions', 'ga:country!=(not set);ga:pagePath==' . $filter, 5, 'analytify-email-country-stats' );
				$cities    = $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:country,ga:city', '-ga:sessions', 'ga:city!=(not set);ga:pagePath==' . $filter, 5, 'analytify-email-city-stats' );
				if ( ! function_exists( 'pa_geopgraphic_lisiting' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/geographic-deprecated.php';
				}
				$message .= pa_geopgraphic_lisiting( $wp_analytify, $countries, $cities );
			}
		}

		// System stats.
		if ( in_array( 'show-system', $selected_sections, true ) ) {
			if ( 'ga4' === $ga_mode ) {
				$browser_stats = $wp_analytify->get_reports(
					'show-default-browser-dashboard',
					array(
						'sessions',
					),
					array(
						'start' => $dates['start_date'],
						'end'   => $dates['end_date'],
					),
					array(
						'browser',
						'operatingSystem',
						'pagePath',
					),
					array(
						'type'  => 'metric',
						'name'  => 'sessions',
						'order' => 'desc',
					),
					array(
						'logic'   => 'AND',
						'filters' => array(
							array(
								'type'           => 'dimension',
								'name'           => 'operatingSystem',
								'match_type'     => 4,
								'value'          => '(not set)',
								'not_expression' => true,
							),
							array(
								'type'           => 'dimension',
								'name'           => 'pagePath',
								'match_type'     => 1,
								'value'          => $filter,
							),
						),
					),
					5
				);

				$os_stats = $wp_analytify->get_reports(
					'show-default-os-dashboard',
					array(
						'sessions',
					),
					array(
						'start' => $dates['start_date'],
						'end'   => $dates['end_date'],
					),
					array(
						'operatingSystem',
						'operatingSystemVersion',
						'pagePath',
					),
					array(
						'type'  => 'metric',
						'name'  => 'sessions',
						'order' => 'desc',
					),
					array(
						'logic'   => 'AND',
						'filters' => array(
							array(
								'type'           => 'dimension',
								'name'           => 'operatingSystemVersion',
								'match_type'     => 4,
								'value'          => '(not set)',
								'not_expression' => true,
							),
							array(
								'type'           => 'dimension',
								'name'           => 'pagePath',
								'match_type'     => 1,
								'value'          => $filter,
							),
						),
					),
					5
				);

				$mobile_stats = $wp_analytify->get_reports(
					'show-default-mobile-dashboard',
					array(
						'sessions',
					),
					array(
						'start' => $dates['start_date'],
						'end'   => $dates['end_date'],
					),
					array(
						'mobileDeviceBranding',
						'mobileDeviceModel',
						'pagePath',
					),
					array(
						'type'  => 'metric',
						'name'  => 'sessions',
						'order' => 'desc',
					),
					array(
						'logic'   => 'AND',
						'filters' => array(
							array(
								'type'           => 'dimension',
								'name'           => 'deviceCategory',
								'match_type'     => 4,
								'value'          => 'desktop',
								'not_expression' => true,
							),
							array(
								'type'           => 'dimension',
								'name'           => 'mobileDeviceModel',
								'match_type'     => 4,
								'value'          => '(not set)',
								'not_expression' => true,
							),
							array(
								'type'           => 'dimension',
								'name'           => 'pagePath',
								'match_type'     => 1,
								'value'          => $filter,
							),
						),
					),
					5
				);

				if ( ! function_exists( 'pa_system_stats' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/system-stats.php';
					$message .= pa_system_stats( $wp_analytify, $browser_stats, $os_stats, $mobile_stats );
				}
			} else {
				$browsers          = $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:browser', '-ga:sessions', 'ga:pagePath==' . $filter, 5, 'analytify-email-browser-stats' );
				$operating_systems = $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:operatingSystem', '-ga:sessions', 'ga:operatingSystem!=(not set);ga:pagePath==' . $filter , 5, 'analytify-email-system-stats' );
				$mobile_phones     = $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:mobileDeviceBranding,ga:mobileDeviceModel', '-ga:sessions', 'ga:mobileDeviceModel!=(not set);ga:mobileDeviceBranding!=(not set);ga:pagePath==' . $filter, 5, 'analytify-email-mobile-stats' );

				if ( ! function_exists( 'pa_system_stats' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/system-stats-deprecated.php';
					$message .= pa_system_stats( $wp_analytify, $browsers, $operating_systems, $mobile_phones );
				}
			}
		}

		/*
		 * Keywords and/or refferals
		 *
		 */
		// check if the 'keywords' are set
		if ( 'ga4' === $ga_mode ) {
			// TODO: add page URL check.
			$keyword_stats_raw = $wp_analytify->get_search_console_stats(
				'analytify-email-keyword-stats',
				array(
					'start' => $dates['start_date'],
					'end'   => $dates['end_date'],
				),
				5
			);

			$keywords = false;
			if ( isset( $keyword_stats_raw['response']['rows'] ) && $keyword_stats_raw['response']['rows'] > 0 ) {
				$keywords = $keyword_stats_raw['response']['rows'];
			}

			$refferals = $wp_analytify->get_reports(
				'analytify-email-refferal-stats',
				array(
					'sessions',
				),
				array(
					'start' => $dates['start_date'],
					'end'   => $dates['end_date'],
				),
				array(
					'sessionSource',
					'sessionMedium',
					'pagePath',
				),
				array(
					'type'  => 'metric',
					'name'  => 'sessions',
					'order' => 'desc',
				),
				array(
					'logic'   => 'AND',
					'filters' => array(
						array(
							'type'       => 'dimension',
							'name'       => 'pagePath',
							'match_type' => 1,
							'value'      => $filter,
						),
					),
				),
				5
			);

			if ( ! function_exists( 'pa_keyword_referral' ) ) {
				include ANALYTIFY_EMAIL_ROOT_PATH . '/views/keyword-referral.php';
				$message .= pa_keyword_referral( $wp_analytify, $keywords, $refferals );
			}
		} else {
			$keywords = ( in_array( 'show-keywords', $selected_sections, true ) ) ? $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:keyword', '-ga:sessions', 'ga:pagePath==' . $filter, 5, 'analytify-email-keyword-stats' ) : false;

			// check if the 'referrer' are set
			$refferals = ( in_array( 'show-referrer', $selected_sections, true ) ) ? $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:source,ga:medium', '-ga:sessions', 'ga:pagePath==' . $filter, 5, 'analytify-email-refferal-stats' ) : false;

			// add stats to the email if either on is set
			if ( $keywords || $refferals ) {
				if ( ! function_exists( 'pa_keyword_referral' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/keyword-referral-deprecated.php';
				}
				$message .= pa_keyword_referral( $wp_analytify, $keywords, $refferals );
			}
		}

		// Social Media.
		if ( in_array( 'show-social', $selected_sections, true ) ) {
			if ( 'ga4' === $ga_mode ) {
				// TODO: missing_ga4
			} else {
				$socials = $wp_analytify->pa_get_analytics_dashboard( 'ga:sessions', $dates['start_date'], $dates['end_date'], 'ga:socialNetwork', '-ga:sessions', 'ga:socialNetwork!=(not set);ga:pagePath==' . $filter, 5, 'analytify-email-social-media-stats' );
				if ( ! function_exists( 'social_stats' ) ) {
					include ANALYTIFY_EMAIL_ROOT_PATH . '/views/social-media.php';
				}
				$message .= social_stats( $wp_analytify, $socials );
			}
		}

		return $message;
	}

	/**
	 * Add Email settings in diagnostic logs.
	 */
	public function analytify_settings_logs() {
		echo "Analytify Email Setting:\r\n";

		$analytify_email = get_option( 'wp-analytify-email' );
		print_r( $analytify_email );

		echo "\r\n";
	}

	/**
	 * Configure backward compatibility for options tabele.
	 */
	public function backward_compatibility() {
		$_analytify_email = get_option( 'wp-analytify-email' );

		if ( empty( $_analytify_email ) ) {
			$default['analytify_email_user_email'] = get_bloginfo( 'admin_email' );
			$default['analytif_email_cron_time']   = array(
				'week' => 'Monday',
			);

			update_option( 'wp-analytify-email', $default );
		}
	}

	/**
	 * Update your plugin from our own server.
	 *
	 * @since 1.0.0
	 */
	public function plugin_updater() {

		// retrieve our license key from the DB
		$_license_key = trim( get_option( 'analytify_email_license_key' ) );

		// setup the updater
		if ( class_exists( 'ANALYTIFY_SL_Plugin_Updater' ) ) {
			$edd_updater = new ANALYTIFY_SL_Plugin_Updater(
				EMAIL_ANALYTIFY_STORE_URL,
				ANALYTIFY_EMAIL_UPGRADE_PATH,
				array(
					'version' => ANALTYIFY_EMAIL_VERSION,               // current version number
					'license' => $_license_key,        // license key (used get_option above to retrieve from DB)
					'item_id' => EMAIL_ANALYTIFY_PRODUCT_ID,    // name of this plugin
					'author'  => 'Muhammad Adnan', // author of this plugin
				)
			);
		}
	}

	/*
	 * Output Email Notifications license activation key fields/inputs
	 * @access public
	 *
	 */
	public function display_license_form() {

		$license_email = get_option( 'analytify_email_license_key' );
		$status_email  = get_option( 'analytify_email_license_status' );
		?>

		<tr valign="top">
			<th scope="row" valign="top">
			<?php esc_html_e( 'Email Notifications:', 'wp-analytify-email' ); ?>
			</th>

			<?php if ( 'valid' === $status_email ) : ?>

			<td class="email-license-row">

				<?php echo $this->get_formatted_masked_email_license( $license_email ); ?>

			</td>

			<?php else : ?>

			<td class="email-license-row">
			<input id="analytify_email_license_key" name="analytify_email_license_key" type="text" class="regular-text" value="<?php esc_attr_e( $license_email ); ?>" />

			<input type="submit" class="button-secondary" id="analytify_email_license_activate" name="analytify_license_activate" value="<?php esc_html_e( 'Activate License', 'wp-analytify-email' ); ?>"/>

			<br /><p id="email-license-status">
				<?php
				if ( $status_email ) {
					echo $status_email;
				}
				?>
			</p>

			</td>

			<?php endif; ?>

		</tr>

		<?php
	}

	/**
	 *
	 * Send Ajax on Active license.
	 *
	 * @since 1.0.0
	 */
	public function ajax_activate_license() {

		$this->check_ajax_referer( 'activate-license' );

		$key_rules = array(
			'action'            => 'key',
			'email_license_key' => 'string',
			'context'           => 'key',
			'nonce'             => 'key',
		);
		$this->set_post_data( $key_rules );

		$api_params = array(
			'edd_action' => 'activate_license',
			'license'    => $this->state_data['email_license_key'],
			'item_id'    => EMAIL_ANALYTIFY_PRODUCT_ID, // the name of our product in EDD
			'url'        => home_url(),
		);

		$response = wp_remote_get(
			esc_url_raw( add_query_arg( $api_params, EMAIL_ANALYTIFY_STORE_URL ) ),
			array(
				'timeout'   => 30,
				'sslverify' => false,
			)
		);

		// make sure the response came back okay
		if ( is_wp_error( $response ) ) {
			return false;
		}

		$license_response = json_decode( wp_remote_retrieve_body( $response ) );

		if ( 'valid' === $license_response->license ) {
			$this->set_email_license_key( $this->state_data['email_license_key'] );
			$license_response->masked_license = $this->get_formatted_masked_email_license( $this->state_data['email_license_key'] );
		} else {
			if ( 'invalid' === $license_response->license ) {
				set_site_transient( 'wpanalytify_email_license_response', $license_response, $this->transient_timeout );
				$license_response->error = $this->get_email_license_status_message( $license_response, $this->state_data['context'] );
			}
		}
		// print_r($license_response);
		// wp_die();
		// $license_response->license will be either "valid" or "invalid"
		update_option( 'analytify_email_license_status', $license_response->license );
		update_option( 'analytify_email_license_key', $this->state_data['email_license_key'] );

		$result = $this->end_ajax( json_encode( $license_response ) );

		return $result;
	}

}
