<?php

/**
 * Forms dashboard main class.
 *
 * @since 1.0.0
 */
class Analytify_Forms extends WP_Analytify_Pro_Base {

	public function __construct() {

		parent::__construct();
		$this->_hooks();
		$this->setup_constants();
		$this->includes();
		// $this->scripts();
	}

	public function _hooks() {

		add_action( 'analytify_add_submenu', array( $this, 'add_menu' ), 50 );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'analytify_forms_output_after_script' ), 99999 );
		add_action( 'admin_init', array( $this, 'wp_analytify_forms_register_option' ) );
		add_action( 'admin_init', array( $this, '_plugin_updater' ), 0 );

		add_action( 'wp_analytify_forms_license_key', array( $this, 'display_license_form' ), 11 );
		add_action( 'wp_ajax_wpanalytify_forms_activate_license', array( $this, 'ajax_activate_license' ) );

		add_filter( 'wp_analytify_pro_setting_tabs', array( $this, 'settings_tab' ), 20, 1 );
		add_filter( 'wp_analytify_pro_setting_fields', array( $this, 'setting_fields' ), 20, 1 );

		// for modifying the meta links
		add_action( 'plugin_row_meta', array( $this, 'plugin_row_meta' ), 10, 2 );

		add_action( 'analytify_settings_logs', array( $this, 'settings_logs' ) );
	}

	public function includes() {
		include_once ANALYTIFY_FORMS_ROOT_PATH . '/classes/analytify-forms-helper.php';
		include_once ANALYTIFY_FORMS_ROOT_PATH . '/classes/analytify-forms-tracking.php';
		include_once ANALYTIFY_FORMS_ROOT_PATH . '/classes/class-analytify-forms-rest-api.php';
	}


	/**
	 * Define constant if not already set
	 *
	 * @param  string      $name
	 * @param  string|bool $value
	 */
	private function define( $name, $value ) {
		if ( ! defined( $name ) ) {
			define( $name, $value );
		}
	}


	/**
	 * Setup plugin constants
	 *
	 * @access      private
	 * @since       1.0.0
	 * @return      void
	 */
	private function setup_constants() {

		// Setting Global Values
		$this->define( 'FORMS_ANALYTIFY_PLUGIN_FILE', __FILE__ );
		$this->define( 'FORMS_ANALYTIFY_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
		$this->define( 'FORMS_ANALYTIFY_PLUGIN_DIR', dirname( __FILE__ ) );
		$this->define( 'FORMS_ANALYTIFY_PRODUCT_NAME', 'Analytify - Forms Addon' );
		$this->define( 'FORMS_ANALYTIFY_PRODUCT_ID', 229559 );
		$this->define( 'FORMS_ANALYTIFY_STORE_URL', 'https://analytify.io' );
	}


	/**
	 * Load Admin Scripts
	 *
	 * @since 1.0
	 */
	public function admin_scripts( $page ) {
		if ( 'analytify_page_analytify-settings' === $page ) {
			wp_enqueue_script( 'analytify_forms_main_js', plugins_url( 'assets/js/main.js', __DIR__ ), array( 'jquery' ), ANALYTIFY_FORMS_VERSION );
		}

		if ( 'analytify_page_analytify-forms' === $page ) {
			wp_enqueue_style( 'analytify-dashboard-forms', plugins_url( 'assets/css/common-dashboard.css', dirname( __FILE__ ) ), array(), ANALYTIFY_FORMS_VERSION );

			wp_enqueue_script( 'analytify-stats-forms', plugins_url( 'assets/js/stats-forms.js', dirname( __FILE__ ) ), array( 'jquery' ), ANALYTIFY_FORMS_VERSION, true );
			wp_localize_script(
				'analytify-stats-forms',
				'analytify_stats_forms',
				array(
					'url'              => esc_url_raw( rest_url( 'wp-analytify/v1/get_forms_report/' ) ),
					'delimiter'        => is_callable( array('WPANALYTIFY_Utils', 'get_delimiter') ) ? WPANALYTIFY_Utils::get_delimiter() : '?',
					'nonce'            => wp_create_nonce( 'wp_rest' ),
					'load_via_ajax'    => true,
					'no_stats_message' => esc_html__( 'No activity during this period.', 'wp-analytify-forms' ),
					'error_message'    => esc_html__( 'Something went wrong. Please try again.', 'wp-analytify-forms' ),
				),
			);
		}
	}


	/**
	 * Enqueue Tracking Scripts.
	 *
	 * @return void
	 */
	public function analytify_forms_output_after_script() {
		
		$manual_ua_code = $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'manual_ua_code', 'wp-analytify-authentication', false );
		$install_ga_code = $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'install_ga_code', 'wp-analytify-profile', 'off' );

		if ( ! $manual_ua_code && 'off' === $install_ga_code ) {
			return;
		}
		
		wp_enqueue_script( 'analytify_forms_tracking', plugins_url( 'assets/js/tracking.js', __DIR__ ), array( 'jquery' ), false, true );
		
		// Localize variable 
		wp_localize_script(
			'analytify_forms_tracking',
			'FORMS_TRACKING_OBJECT',
			array(
				'property_tracking_mode'            => is_callable( array('WPANALYTIFY_Utils', 'get_ga_mode') ) ? WPANALYTIFY_Utils::get_ga_mode( 'profile_for_posts' ) : 'ga3',
				'ANALYTIFY_TRACKING_MODE'			=> ANALYTIFY_TRACKING_MODE,
				'ANALYTIFY_FORMS_CUSTOM_FORM_CLASS'	=> ANALYTIFY_FORMS_CUSTOM_FORM_CLASS,
				'custom_form'						=> $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'custom_form', 'wp-analytify-forms' ),
				'track_cf7'							=> $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'track_cf7', 'wp-analytify-forms' ),
				'track_wpforms'						=> $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'track_wpforms', 'wp-analytify-forms' ),
				'track_gravityforms'				=> $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'track_gravityforms', 'wp-analytify-forms' ),
				'track_formidable'					=> $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'track_formidable', 'wp-analytify-forms' ),
				'track_ninjaforms'					=> $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'track_ninjaforms', 'wp-analytify-forms' ),
				'track_fluentforms'					=> $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'track_fluentforms', 'wp-analytify-forms' ),
			)
		);
	}


	/**
	 * Plugin row meta links
	 * 
	 * @param array  $input already defined meta links
	 * @param string $file plugin file path and name being processed
	 * @return array $input
	 */
	public function plugin_row_meta( $input, $file ) {

		// exit if its other plugins
		if ( 'wp-analytify-forms/wp-analytify-forms.php' !== $file ) { return $input; }

		// remove the 'view details' org link
		unset( $input[2] );

		// add the plugin site link
		$input[] = '<a href="https://analytify.io/add-ons/forms-tracking/" target="_blank">Visit plugin site</a>';

		return $input;

	}


	/**
	 * Add menu for Form dashboard page.
	 * 
	 * @since 1.0.0
	 */
	public static function add_menu() {
		add_submenu_page( 'analytify-dashboard', esc_html__( 'Show Forms Tracking Stats in a Dashboard', 'wp-analytify-forms' ), esc_html__( 'Forms Tracking', 'wp-analytify-forms' ), 'edit_posts', 'analytify-forms', array( __CLASS__, 'analytify_page_file_path' ), 40 );
	}


	/**
	 * Output Analytify for Forms license activation key fields/inputs
	 *
	 * @since 1.0.0
	 */
	public function display_license_form() {

		$license_forms = get_option( 'analytify_forms_license_key' );
		$status_forms  = get_option( 'analytify_forms_license_status' );
		?>

		<tr valign="top">

			<th scope="row" valign="top">
				<?php esc_html_e( 'Forms Tracking:', 'wp-analytify-forms' ); ?>
			</th>

			<?php if ( 'valid' === $status_forms ) { ?>

				<td class="forms-license-row">
					<?php echo $this->get_formatted_masked_forms_license( $license_forms ); ?>
				</td>

			<?php } else { ?>

				<td class="forms-license-row">
					<input id="analytify_forms_license_key" name="analytify_forms_license_key" type="text" class="regular-text" value="<?php esc_attr_e( $license_forms ); ?>" />

					<input type="submit" class="button-secondary" id="analytify_forms_license_activate" name="analytify_license_activate" value="<?php esc_html_e( 'Activate License', 'wp-analytify-forms' ); ?>"/>

					<br /><p id="forms-license-status">
					<?php
					if ( $status_forms ) { echo $status_forms; } ?>
					</p>
				</td>

			<?php } ?>
		</tr>

		<?php
	}


	public function wp_analytify_forms_register_option() {
		// creates our settings in the options table
		register_setting( 'analytify-settings', 'analytify_forms_license_key', array( $this, 'analytify_forms_sanitize_license' ) );
	}

	public function analytify_forms_sanitize_license( $new ) {

		$old = get_option( 'analytify_forms_license_key' );

		if ( $old && $old != $new ) {

			delete_option( 'analytify_forms_license_status' ); // new license has been entered, so must reactivate

		}

		return $new;
	}


	/**
	 *
	 * To autoupdate addon.
	 *
	 * @since 1.0.0
	 */
	public function _plugin_updater() {

		// retrieve our license key from the DB
		$wpa_license_key = trim( get_option( 'analytify_forms_license_key' ) );

		// setup the updater
		if ( class_exists( 'ANALYTIFY_SL_Plugin_Updater' ) ) {
			$edd_updater = new ANALYTIFY_SL_Plugin_Updater(
				FORMS_ANALYTIFY_STORE_URL, 'wp-analytify-forms/wp-analytify-forms.php', array(
					'version' => ANALYTIFY_FORMS_VERSION, // current version number
					'license' => $wpa_license_key, // license key (used get_option above to retrieve from DB)
					'item_id' => FORMS_ANALYTIFY_PRODUCT_ID, // ID of this plugin
					'author'  => 'Muhammad Adnan', // author of this plugin
				)
			);
		}
	}


	public function settings_tab( $old_tabs ) {
		$pro_tabs = array(
			array(
				'id'       => 'wp-analytify-forms',
				'title'    => esc_html__( 'Forms Tracking', 'wp-analytify-forms' ),
				'desc'     => esc_html__( 'Track form conversions in Google Analytics.', 'wp-analytify-forms' ),
				'priority' => '0',
			)
		);

		return array_merge( $old_tabs, $pro_tabs );
	}

	/**
	 * Add settings for form.
	 *
	 * @param array $old_fields Setting Fields passed by the filter.
	 * @return array
	 */
	public function setting_fields( $old_fields ) {

		$pro_fields = array(
			'wp-analytify-forms' => array(

				// checkbox for custom forms
				array(
					'name' 		=> 'custom_form',
					'label'		=> esc_html__( 'Enable Tracking on Custom Forms', 'wp-analytify-forms' ),
					'desc' 		=> esc_html__( 'This will allow Analytify to track custom forms that have the class \''.ANALYTIFY_FORMS_CUSTOM_FORM_CLASS.'\'. Name attribute is also required.', 'wp-analytify-forms' ),
					'type'  	=> 'checkbox',
					'default'	=> 'on',
				)

			),
		);

		// All the supported forms, except custom.
		$supported_forms = AnalytifyFormHelper::get_supported_forms();

		foreach ( $supported_forms as $form ) {

			$option = array(
				'name'  => $form['option_name'],
				// Translators: %s is the name of the 3rd party forms plugin.
				'label' => sprintf( esc_html__( 'Enable Tracking for %s', 'wp-analytify-forms' ), $form['title'] ),
				'type'  => 'checkbox',
			);

			// Translators: %s is the name of the 3rd party forms plugin.
			$option['desc'] = sprintf( esc_html__( 'Enable/Disable from tracking of forms build with %s', 'wp-analytify-forms' ), $form['title'] );

			if ( ! is_plugin_active( $form['plugin_file'] ) ) {
				// Translators: %s is the name of the 3rd party forms plugin.
				$option['desc']                = sprintf( esc_html__( 'You don\'t have the %s plugin activated.', 'wp-analytify-forms' ), $form['title'] );
				$option['options']['disabled'] = true;
			}

			$pro_fields['wp-analytify-forms'][] = $option;
		}

		return array_merge( $old_fields, $pro_fields );
	}


	/**
	 * Include view file.
	 *
	 * @since 1.0.0
	 */
	public static function analytify_page_file_path() {

		$wp_analytify   = $GLOBALS['WP_ANALYTIFY'];
		$start_date_val = strtotime( '- 30 days' );
		$end_date_val   = strtotime( 'now' );
		$start_date     = date( 'Y-m-d', $start_date_val );
		$end_date       = date( 'Y-m-d', $end_date_val );
		$access_token   = get_option( 'post_analytics_token' );

		if ( isset( $_POST['analytify_date_start'] ) && ! empty( $_POST['analytify_date_start'] ) && isset( $_POST['analytify_date_end'] ) && ! empty( $_POST['analytify_date_end'] ) ) {
			$start_date	= sanitize_text_field( wp_unslash( $_POST['analytify_date_start'] ) );
			$end_date	= sanitize_text_field( wp_unslash( $_POST['analytify_date_end'] ) );
		}

		// Fetch Dashboard Profile ID.
		$dashboard_profile_id = $wp_analytify->settings->get_option( 'profile_for_dashboard', 'wp-analytify-profile' );
		$dashboard_profile_id = method_exists( 'WPANALYTIFY_Utils', 'get_reporting_property' ) ? WPANALYTIFY_Utils::get_reporting_property() : $dashboard_profile_id;
		$version              = defined( 'ANALYTIFY_PRO_VERSION' ) ? ANALYTIFY_PRO_VERSION : ANALYTIFY_VERSION;

		/*
		* Check with roles assigned at dashboard settings.
		*/
		$is_access_level = $wp_analytify->settings->get_option( 'show_analytics_roles_dashboard', 'wp-analytify-dashboard' );

		$_analytify_profile = get_option( 'wp-analytify-profile' );

		require_once ANALYTIFY_FORMS_ROOT_PATH . '/views/dashboard-forms.php';

	}


	/**
	 *
	 * Send Ajax on Active license.
	 *
	 * @since 1.0.0
	 */
	public function ajax_activate_license() {

		$this->check_ajax_referer( 'activate-license' );

		$key_rules = array(
			'action'            => 'key',
			'forms_license_key' => 'string',
			'context'           => 'key',
			'nonce'             => 'key',
		);
		$this->set_post_data( $key_rules );

		$api_params = array(
			'edd_action' => 'activate_license',
			'license'    => $this->state_data['forms_license_key'],
			'item_id'    => FORMS_ANALYTIFY_PRODUCT_ID, // the name of our product in EDD
			'url'        => home_url(),
		);

		$response = wp_remote_get(
			esc_url_raw( add_query_arg( $api_params, FORMS_ANALYTIFY_STORE_URL ) ), array(
				'timeout'   => 15,
				'sslverify' => false,
			)
		);

		// make sure the response came back okay
		if ( is_wp_error( $response ) ) {
			// var_dump($response);
			// wp_die();
			return false;
		}

		$license_response = json_decode( wp_remote_retrieve_body( $response ) );

		if ( 'valid' === $license_response->license ) {
			$this->set_forms_license_key( $this->state_data['forms_license_key'] );
			$license_response->masked_license = $this->get_formatted_masked_forms_license( $this->state_data['forms_license_key'] );
		} else {

			if ( 'invalid' === $license_response->license ) {
				// var_dump($license_response->license);
				// wp_die();
				set_site_transient( 'wpanalytify_forms_license_response', $license_response, $this->transient_timeout );
				$license_response->error = $this->get_forms_license_status_message( $license_response, $this->state_data['context'] );

			}
		}

		// $license_response->license will be either "valid" or "invalid"
		update_option( 'analytify_forms_license_status', $license_response->license );
		update_option( 'analytify_forms_license_key', $this->state_data['forms_license_key'] );

		$result = $this->end_ajax( json_encode( $license_response ) );

		return $result;
	}


	/**
	 * Add settings in diagnostic information.
	 *
	 */
	public function settings_logs() {

		echo "\r\n";

		echo "-- Forms Tracking Setting --\r\n \r\n";
		
		$options = get_option( 'wp-analytify-forms' );

		if ( method_exists( 'WPANALYTIFY_Utils', 'print_settings_array' ) ) {
			WPANALYTIFY_Utils::print_settings_array( $options );
		}
	}

	
}

new Analytify_Forms();
