<?php

if ( ! class_exists( 'AnalytifyFormsSubmissionTracking' ) ) {
	/**
	 * Class that is used to make 'submission' calls to the Google Analytics API.
	 * Handles both UA and GA4 calls.
	 */
	class AnalytifyFormsSubmissionTracking {

		/**
		 * The single instance of the class.
		 *
		 * @var object
		 */
		private static $instance;

		/**
		 * GA4 should be tracked.
		 *
		 * @var boolean
		 */
		private $track_ga4;

		/**
		 * Supported forms.
		 *
		 * @var array
		 */
		private $can_track = array( 'cf7', 'wpforms', 'ninjaforms', 'gravityforms', 'formidable', 'fluentforms' );

		/**
		 * Enable/Disable UA Tracking debug.
		 *
		 * @var array
		 */
		private $UA_debug = false;

		/**
		 * Returns the single instance of the class.
		 *
		 * @return object Class instance
		 */
		public static function get_instance() {
			if ( empty( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Class constructor.
		 */
		private function __construct() {

			$this->track_ga4 = method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode();

			$this->hooks();

		}

		/**
		 * Register all the hooks for the class.
		 *
		 * @return void
		 */
		private function hooks() {

			// Contact Form 7.
			add_action( 'wpcf7_before_send_mail', array( $this, 'submission_contact_form_7' ), 15, 1 );

			// WPForms.
			add_action( 'wpforms_process_entry_save', array( $this, 'submission_wpforms' ), 10, 4 );

			// Ninja Forms.
			add_action( 'ninja_forms_after_submission', array( $this, 'submission_ninja_forms' ) );

			// Gravity Forms.
			add_action( 'gform_after_submission', array( $this, 'submission_gravityforms' ), 10, 2 );

			// Formidable Forms.
			add_action ( 'frm_after_create_entry', array( $this, 'submission_formidable' ), 30, 2 );

			// Fluent Forms.
			add_action( 'fluentform_before_submission_confirmation', array( $this, 'submission_fluentforms_save' ), 10, 3 );

		}

		/**
		 * Tracks submission of a contact form 7 form.
		 *
		 * @param object $form_data The form data.
		 *
		 * @return void
		 */
		public function submission_contact_form_7( $form_data ) {
			if ( ! $this->should_track( 'cf7' ) ) {
				return;
			}

			$event = array(
				'category' => 'analytify_form_wpcf7',
				'label'    => (string) $form_data->id(),
			);
			$this->track_form_submission( $event );
		}

		/**
		 * Tracks submission of a WPForms form.
		 *
		 * @param array $fields    The form fields.
		 * @param array $entry     The entry data.
		 * @param int   $form_id   The form ID.
		 * @param array $form_data The form data.
		 *
		 * @return void
		 */
		public function submission_wpforms( $fields, $entry, $form_id, $form_data ) {
			if ( ! $this->should_track( 'wpforms' ) ) {
				return;
			}

			$event = array(
				'category' => 'analytify_form_wpforms',
				'label'    => (string) $form_id,
			);
			$this->track_form_submission( $event );
		}

		/**
		 * Tracks submission of a Ninja Forms.
		 *
		 * @param array $data Form data.
		 *
		 * @return void
		 */
		public function submission_ninja_forms( $data ) {
			if ( ! $this->should_track( 'ninjaforms' ) ) {
				return;
			}

			$event = array(
				'category' => 'analytify_form_ninja_forms',
				'label'    => (string) $data['form_id'],
			);
			$this->track_form_submission( $event );
		}

		/**
		 * Tracks submission of a Gravity Forms.
		 *
		 * @param array $entry The entry data.
		 * @param array $form  The form data.
		 *
		 * @return void
		 */
		public function submission_gravityforms( $entry, $form ) {
			if ( ! $this->should_track( 'gravityforms' ) ) {
				return;
			}

			$event = array(
				'category' => 'analytify_form_gf',
				'label'    => (string) $form['id'],
			);
			$this->track_form_submission( $event );
		}

		/**
		 * Tracks submission of a Formidable Forms.
		 *
		 * @param int $entry_id The entry ID.
		 * @param int $form_id  The form ID.
		 *
		 * @return void
		 */
		public function submission_formidable( $entry_id, $form_id ) {
			if ( ! $this->should_track( 'formidable' ) ) {
				return;
			}

			$event = array(
				'category' => 'analytify_form_formidable',
				'label'    => (string) $form_id,
			);
			$this->track_form_submission( $event );
		}

		/**
		 * Tracks submission of a Formidable Forms.
		 *
		 * @param int    $insertId Form entry insert ID.
		 * @param object $formData Form Entry.
		 * @param object $form     Form Data.
		 * @return void
		 */
		public function submission_fluentforms_save( $insertId, $formData, $form ) {
			if ( ! $this->should_track( 'fluentforms' ) ) {
				return;
			}

			$event = array(
				'category' => 'analytify_form_fluentforms',
				'label'    => (string) $form->id,
			);
			$this->track_form_submission( $event );
		}

		/**
		 * Checks from the settings, if the form should be tracked or not.
		 *
		 * @param string $form Form name.
		 *
		 * @return boolean
		 */
		private function should_track( $form ) {

			// Should this user be tracked?
			if ( function_exists( 'analytify_is_track_user' ) && false === analytify_is_track_user() ) {
				return false;
			}

			// Form is supported?
			if ( ! in_array( $form, $this->can_track, true ) ) {
				return false;
			}

			// Is this form tracking enabled?
			if ( 'on' !== $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'track_' . $form, 'wp-analytify-forms' ) ) {
				return false;
			}

			return true;

		}

		/**
		 * Track form data.
		 * Checks for GA4 or UA tracking.
		 *
		 * @param array $event_data The event data.
		 *
		 * @return void
		 */
		private function track_form_submission( $event_data ) {

			$event_data['uid'] = '';

			$settings = new WP_Analytify_Settings();
			if ( is_user_logged_in() && $settings->get_option( 'track_user_id', 'wp-analytify-advanced' ) ) {
				$event_data['uid'] = get_current_user_id();
			}

			if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {

				// GA4 Tracking.
				$event = array(
					'name'   => 'analytify_form',
					'params' => array(
						'wpa_link_action' => 'submission',
						'wpa_category'    => $event_data['category'],
						'wpa_link_label'  => $event_data['label'],
					),
				);

				if ( $event_data['uid'] ) {
					$event['params']['uid'] = $event_data['uid'];
				}

				analytify_mp_ga4( array( $event ) );

			} else {

				// UA tracking.
				$this->ua_api_call( $event_data );

			}

		}

		/**
		 * Makes the measurement protocol api call for ua.
		 *
		 * @param array $request The request data.
		 *
		 * @return void
		 */
		private function ua_api_call( $request ) {

			// Get user's IP address when possible.
			$ip = '';
			if ( ! empty( $_SERVER['HTTP_CLIENT_IP'] ) && ! filter_var( $_SERVER['HTTP_CLIENT_IP'], FILTER_VALIDATE_IP ) === false ) {
				$ip = $_SERVER['HTTP_CLIENT_IP'];
			} elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) && ! filter_var( $_SERVER['HTTP_X_FORWARDED_FOR'], FILTER_VALIDATE_IP ) === false ) {
				$ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
			} else {
				$ip = $_SERVER['REMOTE_ADDR'];
			}

			$user_language = isset( $_SERVER['HTTP_ACCEPT_LANGUAGE'] ) ? explode( ',', $_SERVER['HTTP_ACCEPT_LANGUAGE'] ) : array();
			$user_language = reset( $user_language );
			$body = array(

				// Change the default type to event.
				't'  => 'event',

				// Required: Event Category.
				'ec' => 'analytify_form',

				// Required: Event Action.
				'ea' => 'submission',

				// Event Category.
				'ec' => $request['category'],

				// Event Label (form title).
				'el' => $request['label'],

				// User ID.
				'uid' => $request['uid'],

				// Optional: Event Value.
				'ev' => 1,

				// Required - Version.
				'v' => '1',

				// Required - UA code.
				'tid' => WP_ANALYTIFY_FUNCTIONS::get_UA_code(),

				// Required - Unique (anonymous) visitor ID.
				'cid' => $this->forms_cid(),

				// Non interaction.
				'ni' => true,

				'dh' => str_replace( array( 'http://', 'https://' ), '', site_url() ),

				'dp' => $_SERVER['REQUEST_URI'],

				'dt' => get_the_title(),

				// Hits that usually also go with JS.
				// Optional - User language.
				'ul' => $user_language,

				// Optional - User IP, to make sure its not the servers'.
				'uip' => $ip,

				// Optional - User Agent.
				'ua' => $_SERVER['HTTP_USER_AGENT'],

				'z' => time(),
			);

			$url = $this->UA_debug ? 'https://www.google-analytics.com/debug/collect' : 'https://ssl.google-analytics.com/collect';

			$response = wp_remote_post(
				$url,
				array(
					'method'   => 'POST',
					'timeout'  => '5',
					'blocking' => false,
					'body'     => $body,
				)
			);


			$check_response =  $response;

		}

		/**
		 * Generates unique (anonymous) visitor ID.
		 *
		 * @return string
		 */
		private function forms_cid() {
			if ( ! empty( $_COOKIE['_ga'] ) ) {
				list( $version, $domainDepth, $cid1, $cid2 ) = preg_split( '[\.]', $_COOKIE['_ga'], 4 );
				$contents = array(
					'version'     => $version,
					'domainDepth' => $domainDepth,
					'cid'         => $cid1 . '.' . $cid2,
				);
				$cid = $contents['cid'];
				return $cid;
			} else {
				// Generate ID.
				return sprintf(
					'%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
					// 32 bits for "time_low"
					mt_rand( 0, 0xffff ), mt_rand( 0, 0xffff ),
					// 16 bits for "time_mid"
					mt_rand( 0, 0xffff ),
					// 16 bits for "time_hi_and_version",
					// four most significant bits holds version number 4
					mt_rand( 0, 0x0fff ) | 0x4000,
					// 16 bits, 8 bits for "clk_seq_hi_res",
					// 8 bits for "clk_seq_low",
					// two most significant bits holds zero and one for variant DCE1.1
					mt_rand( 0, 0x3fff ) | 0x8000,
					// 48 bits for "node"
					mt_rand( 0, 0xffff ), mt_rand( 0, 0xffff ), mt_rand( 0, 0xffff )
				);
			}
		}

	}
}

AnalytifyFormsSubmissionTracking::get_instance();
