<?php
// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handle Analytify REST end points
 */
class Analytify_Forms_Rest_API {

	/**
	 * The single instance of the class.
	 *
	 * @var object
	 */
	private static $instance;

	/**
	 * The main Analytify object.
	 *
	 * @var object
	 */
	private $wp_analytify;

	/**
	 * GA version (ga4 or ga3).
	 *
	 * @var string
	 */
	private $ga_mode;

	/**
	 * Selected 'start state'.
	 *
	 * @var string
	 */
	private $start_date;

	/**
	 * Selected 'End state'.
	 *
	 * @var string
	 */
	private $end_date;

	/**
	 * 'info_message' for forms, its returned with response body.
	 *
	 * @var string
	 */
	private $info_message;

	/**
	 * Returns the single instance of the class.
	 *
	 * @return object Class instance
	 */
	public static function get_instance() {
		if ( empty( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Class constructor.
	 */
	private function __construct() {

		// Register API endpoints.
		add_action( 'rest_api_init', array( $this, 'rest_api_init' ) );

	}

	/**
	 * Register end point.
	 */
	public function rest_api_init() {

		$this->wp_analytify = $GLOBALS['WP_ANALYTIFY'];
		$this->ga_mode      = method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) ? WPANALYTIFY_Utils::get_ga_mode() : 'ga3';

		register_rest_route(
			'wp-analytify/v1',
			'/get_forms_report/(?P<request_type>[a-zA-Z0-9-]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE, // Get Request.
					'callback'            => array( $this, 'handle_request' ),
					'permission_callback' => array( $this, 'permission_check' ),
				),
			)
		);
	}

	/**
	 * Checks access permission.
	 * Checks if the user is logged-in and checks of the user role has access.
	 *
	 * @return boolean
	 */
	public function permission_check() {
		$is_access_level = $this->wp_analytify->settings->get_option( 'show_analytics_roles_dashboard', 'wp-analytify-dashboard', array( 'administrator' ) );
		return (bool) $this->wp_analytify->pa_check_roles( $is_access_level );
	}

	/**
	 * Handle the request.
	 *
	 * @param WP_REST_Request $request WP Request object.
	 *
	 * @return array|WP_Error
	 */
	public function handle_request( WP_REST_Request $request ) {

		$request_type = $request->get_param( 'request_type' );

		$this->start_date = $request->get_param( 'sd' );
		$this->end_date   = $request->get_param( 'ed' );

		$this->info_message = esc_html__( '\'Impressions\' show how many times a particular form has been viewed.', 'wp-analytify-forms' );

		switch ( $request_type ) {
			case 'main-dashboard':
				$forms = 'ga4' === $this->ga_mode ? $this->main_dashboard_ga4() : $this->main_dashboard_ua();
				return array(
					'success' => is_array( $forms ),
					'headers' => array(
						'id'         => array(
							'label'    => esc_html__( 'ID', 'wp-analytify-forms' ),
							'th_class' => 'analytify_num_row',
							'td_class' => 'analytify_txt_center',
						),
						'title'      => array(
							'label'    => esc_html__( 'Name', 'wp-analytify-forms' ),
							'th_class' => 'analytify_txt_left',
							'td_class' => '',
						),
						'impression' => array(
							'label'    => esc_html__( 'Impressions', 'wp-analytify-forms' ),
							'th_class' => 'analytify_value_row',
							'td_class' => 'analytify_txt_center',
						),
						'submission' => array(
							'label'    => esc_html__( 'Submissions', 'wp-analytify-forms' ),
							'th_class' => 'analytify_value_row',
							'td_class' => 'analytify_txt_center',
						),
					),
					'stats'   => $forms,
				);
		}

		// If no request type match, Return error.
		return new WP_Error( 'analytify_invalid_endpoint', esc_html__( 'Invalid endpoint.', 'wp-analytify' ), array( 'status' => 404 ) );
	}

	/**
	 * Returns formatted stats for dashboard of a GA4 property.
	 *
	 * @return array
	 */
	private function main_dashboard_ga4() {

		// API request limit.
		$api_request_limit = apply_filters( 'analytify_api_limit_forms_addon', 100, $this->ga_mode );

		// Data container.
		$stats = array();

		// Build filters for api call and $stats base, based on forms that are available for dashboard.
		$report_filter = array();
		foreach ( AnalytifyFormHelper::get_forms_for_dashboard() as $form ) {

			// API filter.
			$report_filter[] = array(
				'type'       => 'dimension',
				'name'       => 'customEvent:wpa_category',
				'match_type' => 1,
				'value'      => $form['category'],
			);

			// Stats base.
			$stats[ $form['category'] ] = array(
				'stats'        => array(),
				'info_message' => $this->info_message,
			);
		}

		// Fetch report.
		$raw_report = $this->wp_analytify->get_reports(
			'show-analytify-forms-dashboard',
			array(
				'eventCount',
			),
			array(
				'start' => $this->start_date,
				'end'   => $this->end_date,
			),
			array(
				'customEvent:wpa_category',
				'customEvent:wpa_link_action',
				'customEvent:wpa_link_label',
			),
			array(
				'type'  => 'metric',
				'name'  => 'eventCount',
				'order' => 'desc',
			),
			array(
				'logic'   => 'OR',
				'filters' => $report_filter,
			),
			$api_request_limit,
			true,
		);

		// Break the report into separate forms for submission and impression.
		if ( isset( $raw_report['rows'] ) && ! empty( $raw_report['rows'] ) ) {
			foreach ( $raw_report['rows'] as $raw_rows ) {

				$_ct   = $raw_rows['customEvent:wpa_category'];
				$_lb   = $raw_rows['customEvent:wpa_link_label'];
				$_ac   = $raw_rows['customEvent:wpa_link_action'];
				$count = $raw_rows['eventCount'];

				$stats = apply_filters( 'analytify_forms_row_format', $stats, $_ct, $_lb, $_ac, $count );

			}
		}

		return $stats;
	}

	/**
	 * Returns formatted stats for dashboard of a UA property.
	 *
	 * @return array
	 */
	private function main_dashboard_ua() {

		// API request limit.
		$api_request_limit = apply_filters( 'analytify_api_limit_forms_addon', 50 );

		// Data container.
		$stats = array();

		// Build filters for api call and $stats base, based on forms that are available for dashboard.
		$dimension_filter = array();
		foreach ( AnalytifyFormHelper::get_forms_for_dashboard() as $form ) {
			$dimension_filter[] = 'ga:eventCategory=@' . $form['category'];

			// Stats base.
			$stats[ $form['category'] ] = array(
				'stats'        => array(),
				'info_message' => $this->info_message,
			);
		}

		// Fetch report.
		$raw_report = $this->wp_analytify->pa_get_analytics_dashboard( 'ga:totalEvents', $this->start_date, $this->end_date, 'ga:eventCategory,ga:eventAction,ga:eventLabel', '-ga:totalEvents', implode( ',', $dimension_filter ), $api_request_limit, 'show-analytify-forms-dashboard' );

		// Break the report into separate forms for submission and impression.
		if ( isset( $raw_report['rows'] ) && ! empty( $raw_report['rows'] ) ) {
			foreach ( $raw_report['rows'] as $raw_rows ) {

				$_ct   = $raw_rows[0];
				$_lb   = $raw_rows[2];
				$_ac   = $raw_rows[1];
				$count = $raw_rows[3];

				$stats = apply_filters( 'analytify_forms_row_format', $stats, $_ct, $_lb, $_ac, $count );

			}
		}

		return $stats;
	}
}

/**
 * Init the instance.
 *
 */
Analytify_Forms_Rest_API::get_instance();
