<?php

/**
 * Forms dashboard main class.
 *
 * @since 1.0.0
 */
class Analytify_Addon_Forms extends WP_Analytify_Pro_Base {

	public function __construct() {

		parent::__construct();
		$this->_hooks();
		$this->setup_constants();
		$this->includes();
		// $this->scripts();
	}

	public function _hooks() {

		add_action( 'analytify_add_submenu', array( $this, 'add_menu' ), 50 );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'analytify_forms_output_after_script' ), 99999 );
		add_filter( 'wp_analytify_pro_setting_tabs', array( $this, 'settings_tab' ), 20, 1 );
		add_filter( 'wp_analytify_pro_setting_fields', array( $this, 'setting_fields' ), 20, 1 );

		// for modifying the meta links
		add_action( 'plugin_row_meta', array( $this, 'plugin_row_meta' ), 10, 2 );

		add_action( 'analytify_settings_logs', array( $this, 'settings_logs' ) );
	}

	public function includes() {
		include_once ANALYTIFY_FORMS_ADDON_ROOT_PATH . '/classes/analytify-forms-helper.php';
		include_once ANALYTIFY_FORMS_ADDON_ROOT_PATH . '/classes/analytify-forms-tracking.php';
		include_once ANALYTIFY_FORMS_ADDON_ROOT_PATH . '/classes/class-analytify-forms-rest-api.php';
	}


	/**
	 * Define constant if not already set
	 *
	 * @param  string      $name
	 * @param  string|bool $value
	 */
	private function define( $name, $value ) {
		if ( ! defined( $name ) ) {
			define( $name, $value );
		}
	}


	/**
	 * Setup plugin constants
	 *
	 * @access      private
	 * @since       1.0.0
	 * @return      void
	 */
	private function setup_constants() {

		// Setting Global Values
		$this->define( 'FORMS_ANALYTIFY_PLUGIN_FILE', __FILE__ );
		$this->define( 'FORMS_ANALYTIFY_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
		$this->define( 'FORMS_ANALYTIFY_PLUGIN_DIR', dirname( __FILE__ ) );
		$this->define( 'FORMS_ANALYTIFY_PRODUCT_NAME', 'Analytify - Forms Addon' );
		$this->define( 'FORMS_ANALYTIFY_PRODUCT_ID', 229559 );
		$this->define( 'FORMS_ANALYTIFY_STORE_URL', 'https://analytify.io' );
	}


	/**
	 * Load Admin Scripts
	 *
	 * @since 1.0
	 */
	public function admin_scripts( $page ) {
		if ( 'analytify_page_analytify-settings' === $page ) {
			// wp_enqueue_script( 'analytify_forms_main_js', plugins_url( 'assets/js/main.js', __DIR__ ), array( 'jquery' ), ANALYTIFY_FORMS_ADDON_VERSION );
		}

		if ( 'analytify_page_analytify-forms' === $page ) {
			wp_enqueue_style( 'analytify-dashboard-forms', plugins_url( 'assets/css/common-dashboard.css', dirname( __FILE__ ) ), array(), ANALYTIFY_FORMS_ADDON_VERSION );

			// wp_enqueue_script( 'analytify-stats-forms', plugins_url( 'assets/js/stats-forms.js', dirname( __FILE__ ) ), array( 'jquery' ), ANALYTIFY_FORMS_ADDON_VERSION, true );
			wp_localize_script(
				'analytify-stats-forms',
				'analytify_stats_forms',
				array(
					'url'              => esc_url_raw( rest_url( 'wp-analytify/v1/get_forms_report/' ) ),
					'delimiter'        => is_callable( array('WPANALYTIFY_Utils', 'get_delimiter') ) ? WPANALYTIFY_Utils::get_delimiter() : '?',
					'nonce'            => wp_create_nonce( 'wp_rest' ),
					'load_via_ajax'    => true,
					'no_stats_message' => esc_html__( 'No activity during this period.', 'wp-analytify-forms' ),
					'error_message'    => esc_html__( 'Something went wrong. Please try again.', 'wp-analytify-forms' ),
				),
			);
		}
	}


	/**
	 * Enqueue Tracking Scripts.
	 *
	 * @return void
	 */
	public function analytify_forms_output_after_script() {
		
		$manual_ua_code = $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'manual_ua_code', 'wp-analytify-authentication', false );
		$install_ga_code = $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'install_ga_code', 'wp-analytify-profile', 'off' );

		if ( ! $manual_ua_code && 'off' === $install_ga_code ) {
			return;
		}
		
		wp_enqueue_script( 'analytify_forms_tracking', plugins_url( 'assets/js/tracking.js', __DIR__ ), array( 'jquery' ), false, true );
		
		// Localize variable 
		wp_localize_script(
			'analytify_forms_tracking',
			'FORMS_TRACKING_OBJECT',
			array(
				'property_tracking_mode'            		=> is_callable( array('WPANALYTIFY_Utils', 'get_ga_mode') ) ? WPANALYTIFY_Utils::get_ga_mode( 'profile_for_posts' ) : 'ga3',
				'ANALYTIFY_TRACKING_MODE'					=> ANALYTIFY_TRACKING_MODE,
				'ANALYTIFY_FORMS_ADDON_CUSTOM_FORM_CLASS'	=> ANALYTIFY_FORMS_ADDON_CUSTOM_FORM_CLASS,
				'custom_form'								=> $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'custom_form', 'wp-analytify-forms' ),
				'track_cf7'									=> $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'track_cf7', 'wp-analytify-forms' ),
				'track_wpforms'								=> $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'track_wpforms', 'wp-analytify-forms' ),
				'track_gravityforms'						=> $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'track_gravityforms', 'wp-analytify-forms' ),
				'track_formidable'							=> $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'track_formidable', 'wp-analytify-forms' ),
				'track_ninjaforms'							=> $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'track_ninjaforms', 'wp-analytify-forms' ),
				'track_fluentforms'							=> $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'track_fluentforms', 'wp-analytify-forms' ),
			)
		);
	}


	/**
	 * Plugin row meta links
	 * 
	 * @param array  $input already defined meta links
	 * @param string $file plugin file path and name being processed
	 * @return array $input
	 */
	public function plugin_row_meta( $input, $file ) {

		// exit if its other plugins
		if ( 'wp-analytify-forms/wp-analytify-forms.php' !== $file ) { return $input; }

		// remove the 'view details' org link
		unset( $input[2] );

		// add the plugin site link
		$input[] = '<a href="https://analytify.io/add-ons/forms-tracking/" target="_blank">Visit plugin site</a>';

		return $input;

	}


	/**
	 * Add menu for Form dashboard page.
	 * 
	 * @since 1.0.0
	 */
	public static function add_menu() {
		add_submenu_page( 'analytify-dashboard', esc_html__( 'Show Forms Tracking Stats in a Dashboard', 'wp-analytify-forms' ), esc_html__( 'Forms Tracking', 'wp-analytify-forms' ), 'edit_posts', 'analytify-forms', array( __CLASS__, 'analytify_page_file_path' ), 40 );
	}

	public function wp_analytify_forms_register_option() {
		// creates our settings in the options table
		register_setting( 'analytify-settings', 'analytify_forms_license_key', array( $this, 'analytify_forms_sanitize_license' ) );
	}

	public function analytify_forms_sanitize_license( $new ) {

		$old = get_option( 'analytify_forms_license_key' );

		if ( $old && $old != $new ) {

			delete_option( 'analytify_forms_license_status' ); // new license has been entered, so must reactivate

		}

		return $new;
	}

	public function settings_tab( $old_tabs ) {
		$pro_tabs = array(
			array(
				'id'       => 'wp-analytify-forms',
				'title'    => esc_html__( 'Forms Tracking', 'wp-analytify-forms' ),
				'desc'     => esc_html__( 'Track form conversions in Google Analytics.', 'wp-analytify-forms' ),
				'priority' => '0',
			)
		);

		return array_merge( $old_tabs, $pro_tabs );
	}

	/**
	 * Add settings for form.
	 *
	 * @param array $old_fields Setting Fields passed by the filter.
	 * @return array
	 */
	public function setting_fields( $old_fields ) {

		$pro_fields = array(
			'wp-analytify-forms' => array(

				// checkbox for custom forms
				array(
					'name' 		=> 'custom_form',
					'label'		=> esc_html__( 'Enable Tracking on Custom Forms', 'wp-analytify-forms' ),
					'desc' 		=> esc_html__( 'This will allow Analytify to track custom forms that have the class \''.ANALYTIFY_FORMS_ADDON_CUSTOM_FORM_CLASS.'\'. Name attribute is also required.', 'wp-analytify-forms' ),
					'type'  	=> 'checkbox',
					'default'	=> 'off',
				)

			),
		);

		// All the supported forms, except custom.
		$supported_forms = AnalytifyAddonFormHelper::get_supported_forms();

		foreach ( $supported_forms as $form ) {

			$option = array(
				'name'  => $form['option_name'],
				// Translators: %s is the name of the 3rd party forms plugin.
				'label' => sprintf( esc_html__( 'Enable Tracking for %s', 'wp-analytify-forms' ), $form['title'] ),
				'type'  => 'checkbox',
			);

			// Translators: %s is the name of the 3rd party forms plugin.
			$option['desc'] = sprintf( esc_html__( 'Enable/Disable from tracking of forms build with %s', 'wp-analytify-forms' ), $form['title'] );

			if ( ! is_plugin_active( $form['plugin_file'] ) ) {

				// Check added for wp forms.
				if ( $form['title'] === 'WPForms' && is_plugin_active( 'wpforms/wpforms.php' ) ) {
					// intentionally not doing anything
				} else {
					// Translators: %s is the name of the 3rd party forms plugin.
					$option['desc']                = sprintf( esc_html__( 'You don\'t have the %s plugin activated.', 'wp-analytify-forms' ), $form['title'] );
					$option['options']['disabled'] = true;
				}
			}

			$pro_fields['wp-analytify-forms'][] = $option;
		}

		return array_merge( $old_fields, $pro_fields );
	}

	/**
	 * Include view file.
	 *
	 * @since 1.0.0
	 */
	public static function analytify_page_file_path() {

		$wp_analytify   = $GLOBALS['WP_ANALYTIFY'];
		$access_token   = get_option( 'post_analytics_token' );

		//Get the start date and end date from wpa-core-functions
		if ( function_exists( 'analytify_datepicker_dates' ) ) {
			extract( analytify_datepicker_dates() );
		} else {
			$start_date = wp_date( 'Y-m-d', strtotime( '-1 month' ) );
			$end_date   = wp_date( 'Y-m-d', strtotime( 'now' ) );
		}

		// Fetch Dashboard Profile ID.
		$dashboard_profile_id = $wp_analytify->settings->get_option( 'profile_for_dashboard', 'wp-analytify-profile' );
		$dashboard_profile_id = method_exists( 'WPANALYTIFY_Utils', 'get_reporting_property' ) ? WPANALYTIFY_Utils::get_reporting_property() : $dashboard_profile_id;
		$version              = defined( 'ANALYTIFY_PRO_VERSION' ) ? ANALYTIFY_PRO_VERSION : ANALYTIFY_VERSION;

		/*
		* Check with roles assigned at dashboard settings.
		*/
		$is_access_level = $wp_analytify->settings->get_option( 'show_analytics_roles_dashboard', 'wp-analytify-dashboard' );

		$_analytify_profile = get_option( 'wp-analytify-profile' );

		require_once ANALYTIFY_FORMS_ADDON_ROOT_PATH . '/views/dashboard-forms.php';

	}

	/**
	 * Add settings in diagnostic information.
	 *
	 */
	public function settings_logs() {

		echo "\r\n";

		echo "-- Forms Tracking Setting --\r\n \r\n";
		
		$options = get_option( 'wp-analytify-forms' );

		if ( method_exists( 'WPANALYTIFY_Utils', 'print_settings_array' ) ) {
			WPANALYTIFY_Utils::print_settings_array( $options );
		}
	}

	
}

new Analytify_Addon_Forms();
