<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Analytify_Goals_Loader {

	/**
	 * Singleton instance
	 *
	 * @var Analytify_Goals_Loader
	 */
	private static $instance = null;

	/**
	 * Plugin constants
	 */
	const VERSION = '6.1.0';
	const ROOT_PATH = __DIR__;

	/**
	 * Get singleton instance
	 *
	 * @return Analytify_Goals_Loader
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		$this->define_constants();
		$this->init_hooks();
	}
	
	/**
	 * Define constants
	 */
	private function define_constants() {
		define( 'ANALYTIFY_GOALS_ADDON_VERSION', self::VERSION );
		define( 'ANALYTIFY_GOALS_ADDON_ROOT_PATH', self::ROOT_PATH );
	}

	/**
	 * Initialize hooks
	 */
	private function init_hooks() {
		add_action( 'plugins_loaded', [ $this, 'load_analytify_goals' ], 20 );
	}

	/**
	 * Load Analytify Goals Addon
	 */
	public function load_analytify_goals() {
		$prevent = false;

		if ( ! file_exists( WP_PLUGIN_DIR . '/wp-analytify-pro/wp-analytify-pro.php' ) ) {
			add_action( 'admin_notices', [ $this, 'check_analytify_pro_notice' ] );
			$prevent = true;
		} elseif ( ! class_exists( 'WP_Analytify_Pro_Base' ) ) {
			add_action( 'admin_notices', [ $this, 'activate_analytify_pro_notice' ] );
			$prevent = true;
		}

		if ( $prevent ) {
			add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_notice_styles' ] );
			return;
		}

		include_once ANALYTIFY_GOALS_ADDON_ROOT_PATH . '/class-wp-analytify-goals.php';
	}

	/**
	 * Admin notice: Install Analytify Pro
	 */
	public function check_analytify_pro_notice() {
		$message = __( '<strong>Attention:</strong> Please Install Analytify Pro to use Analytify Goals addon.', 'wp-analytify-goals' );
		$this->display_notice( $message, 'wp-analytify-danger' );
	}

	/**
	 * Admin notice: Activate Analytify Pro
	 */
	public function activate_analytify_pro_notice() {
		$message = __( '<strong>Attention:</strong> Please activate Analytify Pro to use Analytify Goals addon.', 'wp-analytify-goals' );
		$this->display_notice( $message, 'wp-analytify-danger' );
	}

	/**
	 * Display custom admin notice
	 *
	 * @param string $message Custom message.
	 * @param string $class   Notification class (success, danger, etc.).
	 */
	private function display_notice( $message, $class ) {
		// Only show notices on allowed pages
		$screen = get_current_screen();
		
		if ( ! $screen ) {
			return;
		}
		
		$allowed_parent_bases = array(
			'analytify-dashboard',
			'plugins',
			'analytify-settings',
			'analytify-license',
			'analytify-addons',
			'dashboard',
		);
		
		$allowed_screen_ids = array(
			'plugins',
			'dashboard',
		);
		
		// Check if we're on an allowed page.
		$is_allowed = in_array( $screen->parent_base, $allowed_parent_bases, true )
			|| in_array( $screen->id, $allowed_screen_ids, true )
			|| ( isset( $_GET['page'] ) && strpos( $_GET['page'], 'analytify' ) === 0 );
		
		// Don't show notice if not on allowed page.
		if ( ! $is_allowed ) {
			return;
		}
		
		echo '<div class="wp-analytify-notification ' . esc_attr( $class ) . '">
			<div class="wp-analytify-notice-logo">
				<img src="' . esc_url( plugins_url( 'assets/img/logo.svg', __FILE__ ) ) . '" alt="Analytify Logo">
			</div>
			<div class="wp-analytify-notice-description">
				<p>' . wp_kses_post( $message ) . '</p>
			</div>
		</div>';
	}

	/**
	 * Enqueue admin notice styles
	 */
	public function enqueue_notice_styles() {
		wp_enqueue_style( 'wpag-notices-css', plugins_url( 'assets/css/notices.css', __FILE__ ), [], ANALYTIFY_GOALS_ADDON_VERSION );
	}
}

// Initialize the singleton
Analytify_Goals_Loader::get_instance();
