<?php
/*
Plugin Name: Sity DB Cleaner
Description: أداة لمسح الجداول غير المستخدمة تقريبًا من قاعدة بيانات ووردبريس بعد إزالة البلجنات. تستخدم منطق ذكي لتخمين الجداول اليتيمة. استخدمها بحذر شديد.
Version: 1.1
Author: Sity Cloud
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * إضافة صفحة في لوحة التحكم / Network Admin
 */
function sity_db_cleaner_add_menu() {
    if ( is_multisite() && is_network_admin() ) {
        add_menu_page(
            'منظف الجداول',
            'منظف الجداول',
            'manage_network',
            'sity-db-cleaner',
            'sity_db_cleaner_page',
            'dashicons-database',
            90
        );
    } elseif ( ! is_multisite() ) {
        add_management_page(
            'منظف الجداول',
            'منظف الجداول',
            'manage_options',
            'sity-db-cleaner',
            'sity_db_cleaner_page'
        );
    }
}
add_action( 'network_admin_menu', 'sity_db_cleaner_add_menu' );
add_action( 'admin_menu', 'sity_db_cleaner_add_menu' );

/**
 * جلب كل جداول قاعدة البيانات
 */
function sity_db_cleaner_get_all_tables() {
    global $wpdb;
    $tables = $wpdb->get_col( 'SHOW TABLES' );
    return $tables ? $tables : array();
}

/**
 * جداول ووردبريس الأساسية (محميّة)
 */
function sity_db_cleaner_get_core_tables() {
    global $wpdb;

    $prefix = $wpdb->prefix;

    $core_suffixes = array(
        'commentmeta',
        'comments',
        'links',
        'options',
        'postmeta',
        'posts',
        'termmeta',
        'terms',
        'term_relationships',
        'term_taxonomy',
        'usermeta',
        'users'
    );

    $core_tables = array();
    foreach ( $core_suffixes as $suffix ) {
        $core_tables[] = $prefix . $suffix;
    }

    // جداول عامة في المالتي سايت
    $global_tables = array(
        $wpdb->base_prefix . 'blogs',
        $wpdb->base_prefix . 'blogmeta',
        $wpdb->base_prefix . 'site',
        $wpdb->base_prefix . 'sitemeta',
        $wpdb->base_prefix . 'registration_log',
        $wpdb->base_prefix . 'signups',
        $wpdb->base_prefix . 'sitecategories'
    );

    $core_tables = array_merge( $core_tables, $global_tables );

    return array_unique( $core_tables );
}

/**
 * جلب كل البلجنز المتثبتة وعمل Keywords من أسمائها
 */
function sity_db_cleaner_get_plugin_keywords() {
    if ( ! function_exists( 'get_plugins' ) ) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }

    $plugins  = get_plugins();
    $keywords = array(); // slug => [ 'keyword' => 'slug', 'name' => 'Plugin Name' ]

    foreach ( $plugins as $file => $data ) {
        $slug = dirname( $file );
        if ( '.' === $slug ) {
            // لو البلجن في الروت
            $slug = basename( $file, '.php' );
        }

        $slug_clean = strtolower( preg_replace( '/[^a-z0-9_]+/i', '_', $slug ) );
        if ( empty( $slug_clean ) ) {
            continue;
        }

        $keywords[ $slug_clean ] = array(
            'keyword' => $slug_clean,
            'name'    => isset( $data['Name'] ) ? $data['Name'] : $slug_clean,
        );
    }

    return $keywords;
}

/**
 * تخمين الجداول المستخدمة/اليتيمة بناءً على أسماء البلجنز
 */
function sity_db_cleaner_classify_custom_tables( $tables_custom ) {
    $plugin_keywords = sity_db_cleaner_get_plugin_keywords();

    $used   = array(); // جدول متوقع أنه تابع لبلجن حالي
    $orphan = array(); // جدول مش باين تابع لأي بلجن حالي

    foreach ( $tables_custom as $table ) {
        $table_lower = strtolower( $table );
        $matched     = false;
        $matched_plugin = null;

        foreach ( $plugin_keywords as $slug => $info ) {
            // لو اسم الجدول يحتوي على slug البلجن
            if ( false !== strpos( $table_lower, $slug ) ) {
                $matched         = true;
                $matched_plugin  = $info;
                break;
            }
        }

        if ( $matched && $matched_plugin ) {
            $used[] = array(
                'table'  => $table,
                'plugin' => $matched_plugin['name'],
                'slug'   => $matched_plugin['keyword'],
            );
        } else {
            $orphan[] = array(
                'table'  => $table,
            );
        }
    }

    return array(
        'used'   => $used,
        'orphan' => $orphan,
    );
}

/**
 * صفحة البلجن
 */
function sity_db_cleaner_page() {
    if ( ! current_user_can( is_multisite() ? 'manage_network' : 'manage_options' ) ) {
        wp_die( 'غير مسموح لك بالدخول لهذه الصفحة.' );
    }

    global $wpdb;

    $all_tables  = sity_db_cleaner_get_all_tables();
    $core_tables = sity_db_cleaner_get_core_tables();

    $message = '';
    $error   = '';

    // معالجة الفورم عند مسح الجداول
    if ( isset( $_POST['sity_db_cleaner_action'] ) && $_POST['sity_db_cleaner_action'] === 'delete_tables' ) {
        check_admin_referer( 'sity_db_cleaner_delete_tables', 'sity_db_cleaner_nonce' );

        $selected_tables = isset( $_POST['sity_db_cleaner_tables'] ) ? (array) $_POST['sity_db_cleaner_tables'] : array();
        $confirm_text    = isset( $_POST['sity_db_cleaner_confirm'] ) ? trim( $_POST['sity_db_cleaner_confirm'] ) : '';

        if ( empty( $selected_tables ) ) {
            $error = 'لم تقم باختيار أي جداول.';
        } elseif ( $confirm_text !== 'DELETE' ) {
            $error = 'لابد من كتابة كلمة DELETE بالحروف الكبيرة لتأكيد المسح.';
        } else {
            $dropped = array();

            foreach ( $selected_tables as $table ) {
                $table = esc_sql( $table );

                // منع مسح الجداول الأساسية كطبقة أمان إضافية
                if ( in_array( $table, $core_tables, true ) ) {
                    continue;
                }

                // تأكد أنه موجود فعلاً
                if ( in_array( $table, $all_tables, true ) ) {
                    $wpdb->query( "DROP TABLE IF EXISTS `$table`" );
                    $dropped[] = $table;
                }
            }

            if ( ! empty( $dropped ) ) {
                $message = 'تم مسح الجداول التالية بنجاح:<br>' . implode( '<br>', array_map( 'esc_html', $dropped ) );
                // إعادة تحميل قائمة الجداول بعد المسح
                $all_tables = sity_db_cleaner_get_all_tables();
            } else {
                $error = 'لم يتم مسح أي جدول. ربما اخترت جداول أساسية أو جداول غير موجودة.';
            }
        }
    }

    // تقسيم الجداول إلى أساسية / غير أساسية
    $tables_core   = array();
    $tables_custom = array();

    foreach ( $all_tables as $table ) {
        if ( in_array( $table, $core_tables, true ) ) {
            $tables_core[] = $table;
        } else {
            $tables_custom[] = $table;
        }
    }

    // تصنيف الجداول الإضافية إلى مستخدمة / يتيمة
    $classified = sity_db_cleaner_classify_custom_tables( $tables_custom );
    $tables_used   = $classified['used'];   // فيها plugin name
    $tables_orphan = $classified['orphan']; // جداول بدون أي بلجن مطابق

    ?>
    <div class="wrap">
        <h1>منظف الجداول – Sity DB Cleaner</h1>

        <div style="margin-top: 10px; padding: 15px; border-left: 4px solid #d63638; background: #fff3f3;">
            <strong>تحذير شديد:</strong>
            <p>
                هذه الأداة تقوم بمسح جداول بالكامل من قاعدة البيانات. أي مسح هنا لا يمكن التراجع عنه.<br>
                <strong>يجب</strong> أن تأخذ نسخة احتياطية كاملة من قاعدة البيانات قبل استخدام الأداة.
            </p>
        </div>

        <div style="margin-top: 10px; padding: 10px; border-left: 4px solid #2271b1; background:#f0f6fc; max-width:900px;">
            <strong>منطق الأداة:</strong>
            <ul style="margin-top:5px; list-style:disc; padding-left:20px;">
                <li>تتعرف على جداول ووردبريس الأساسية وتحميها من المسح.</li>
                <li>تقرأ كل البلجنز المتثبتة وتحاول تطابق أسماء الجداول مع أسماء البلجنز.</li>
                <li>الجداول التي لا ينطبق عليها أي بلجن حالي تظهر في قسم <strong>الجداول اليتيمة المحتملة</strong>.</li>
                <li>رغم كدا: ما زال مجرد تخمين ذكي، وليس ضمان 100٪.</li>
            </ul>
        </div>

        <?php if ( ! empty( $message ) ) : ?>
            <div style="margin-top: 10px; padding: 10px; border-left: 4px solid #00a32a; background: #ecf9f1;">
                <?php echo wp_kses_post( $message ); ?>
            </div>
        <?php endif; ?>

        <?php if ( ! empty( $error ) ) : ?>
            <div style="margin-top: 10px; padding: 10px; border-left: 4px solid #d63638; background: #fff3f3;">
                <?php echo esc_html( $error ); ?>
            </div>
        <?php endif; ?>

        <form method="post" style="margin-top: 20px;">
            <?php wp_nonce_field( 'sity_db_cleaner_delete_tables', 'sity_db_cleaner_nonce' ); ?>
            <input type="hidden" name="sity_db_cleaner_action" value="delete_tables">

            <!-- جداول أساسية -->
            <h2>جداول ووردبريس الأساسية (محميّة)</h2>
            <p style="color:#666;">هذه الجداول لا يمكن اختيارها للمسح من خلال الأداة لحمايتك من كسر الموقع.</p>
            <table class="widefat striped" style="max-width:900px;">
                <thead>
                    <tr>
                        <th>اسم الجدول</th>
                    </tr>
                </thead>
                <tbody>
                <?php if ( empty( $tables_core ) ) : ?>
                    <tr><td>لا توجد جداول أساسية مطابقة للبريفكس الحالي.</td></tr>
                <?php else : ?>
                    <?php foreach ( $tables_core as $table ) : ?>
                        <tr>
                            <td><code><?php echo esc_html( $table ); ?></code></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
                </tbody>
            </table>

            <!-- جداول متوقع أنها مستخدمة -->
            <h2 style="margin-top:30px;">جداول متوقع أنها مستخدمة حاليًا (تابعة لبلجن متثبت)</h2>
            <p style="color:#666;">
                الجداول دي ظهر في اسمها جزء من اسم بلجن متثبت حاليًا في النظام.<br>
                <strong>لا يُنصح</strong> بمسحها إلا لو أنت متأكد 100٪ أن البلجن نفسه اتحذف أو مش محتاج بياناته.
            </p>

            <table class="widefat striped" style="max-width:900px;">
                <thead>
                    <tr>
                        <th style="width:40px;">مسح؟</th>
                        <th>اسم الجدول</th>
                        <th>متوقع أنه تابع لـ</th>
                    </tr>
                </thead>
                <tbody>
                <?php if ( empty( $tables_used ) ) : ?>
                    <tr><td colspan="3">لا توجد جداول متوقع أنها مرتبطة ببلجن معين.</td></tr>
                <?php else : ?>
                    <?php foreach ( $tables_used as $item ) : ?>
                        <tr>
                            <td>
                                <input type="checkbox" name="sity_db_cleaner_tables[]" value="<?php echo esc_attr( $item['table'] ); ?>">
                            </td>
                            <td><code><?php echo esc_html( $item['table'] ); ?></code></td>
                            <td><?php echo esc_html( $item['plugin'] ); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
                </tbody>
            </table>

            <!-- جداول يتيمة محتملة -->
            <h2 style="margin-top:30px; color:#d63638;">جداول يتيمة محتملة (لا يوجد بلجن مطابق لها)</h2>
            <p style="color:#666;">
                الجداول دي مش لاقيين أي بلجن متثبت حالياً اسمه قريب من اسمها.<br>
                غالباً تبع بلجن كان متثبت قبل كدا واتحذف، أو سكربت خارجي قديم.<br>
                برضه راجع الأسماء كويس قبل المسح.
            </p>

            <table class="widefat striped" style="max-width:900px;">
                <thead>
                    <tr>
                        <th style="width:40px;">مسح؟</th>
                        <th>اسم الجدول</th>
                    </tr>
                </thead>
                <tbody>
                <?php if ( empty( $tables_orphan ) ) : ?>
                    <tr><td colspan="2">مفيش جداول يتيمة متوقعة – كلها شكلها تابعة لبلجنات متثبتة.</td></tr>
                <?php else : ?>
                    <?php foreach ( $tables_orphan as $item ) : ?>
                        <tr>
                            <td>
                                <input type="checkbox" name="sity_db_cleaner_tables[]" value="<?php echo esc_attr( $item['table'] ); ?>">
                            </td>
                            <td><code><?php echo esc_html( $item['table'] ); ?></code></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
                </tbody>
            </table>

            <?php if ( ! empty( $tables_used ) || ! empty( $tables_orphan ) ) : ?>
                <div style="margin-top: 20px; padding: 15px; border:1px solid #f0ad4e; background:#fff8e5; max-width:900px;">
                    <p><strong>تأكيد نهائي:</strong></p>
                    <p>لكي يتم تنفيذ عملية المسح، اكتب كلمة <code>DELETE</code> بالحروف الإنجليزية الكبيرة في الحقل التالي:</p>
                    <p>
                        <input type="text" name="sity_db_cleaner_confirm" style="width:200px;" placeholder="DELETE">
                    </p>
                    <p>
                        <button type="submit" class="button button-primary" onclick="return confirm('هل أنت متأكد أنك تريد مسح الجداول التي اخترتها؟ لا يمكن التراجع عن هذه العملية!');">
                            مسح الجداول المختارة
                        </button>
                    </p>
                </div>
            <?php endif; ?>
        </form>
    </div>
    <?php
}
