<?php
/*
Plugin Name: Sity Cloud Onboarding Light
Description: صفحة Onboarding قوية بتصميم فاتح وزجاجي عند أول تسجيل دخول – إعداد اسم الموقع، الوصف، SEO، اللوجو، الروابط الدائمة، مع إعدادات إضافية مثل نوع الموقع وصفحات أساسية.
Author: Sity Cloud
Version: 1.0
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * عند التفعيل: نحدد إن الأونبوردنج لسه ما اتعملتش
 */
function sity_onb_activate() {
    if ( ! get_option( 'sity_onb_done' ) ) {
        update_option( 'sity_onb_done', 0 );
    }
}
register_activation_hook( __FILE__, 'sity_onb_activate' );

/**
 * إضافة صفحة الأونبوردنج في منيو الأدمن
 */
function sity_onb_add_menu() {
    add_menu_page(
        'تهيئة موقعك',
        'تهيئة الموقع',
        'manage_options',
        'sity-onboarding',
        'sity_onb_render_page',
        'dashicons-admin-site',
        2
    );
}
add_action( 'admin_menu', 'sity_onb_add_menu' );

/**
 * ريديراكت أول مرة الأدمن يدخل الداشبورد → صفحة الأونبوردنج
 */
function sity_onb_redirect_first_login() {
    if ( is_network_admin() ) {
        return;
    }

    if ( ! is_user_logged_in() ) {
        return;
    }

    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    $done = get_option( 'sity_onb_done', 0 );
    if ( $done ) {
        return;
    }

    $current_page = isset( $_GET['page'] ) ? sanitize_text_field( $_GET['page'] ) : '';
    if ( $current_page === 'sity-onboarding' ) {
        return;
    }

    $screen = get_current_screen();
    if ( $screen && $screen->base === 'dashboard' ) {
        wp_safe_redirect( admin_url( 'admin.php?page=sity-onboarding' ) );
        exit;
    }
}
add_action( 'current_screen', 'sity_onb_redirect_first_login' );

/**
 * إنشاء صفحات أساسية لو المستخدم اختار كده
 */
function sity_onb_maybe_create_basic_pages() {
    $pages_to_create = array(
        'من نحن'          => 'about',
        'تواصل معنا'      => 'contact',
        'سياسة الخصوصية'  => 'privacy-policy',
    );

    foreach ( $pages_to_create as $title => $slug ) {
        $existing = get_page_by_path( $slug );
        if ( ! $existing ) {
            wp_insert_post( array(
                'post_title'   => $title,
                'post_name'    => $slug,
                'post_status'  => 'publish',
                'post_type'    => 'page',
                'post_content' => '',
            ) );
        }
    }
}

/**
 * معالجة الفورم: حفظ البيانات
 */
function sity_onb_handle_post() {
    if ( ! is_admin() ) {
        return;
    }

    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    if ( ! isset( $_POST['sity_onb_submit'] ) ) {
        return;
    }

    check_admin_referer( 'sity_onb_form', 'sity_onb_nonce' );

    // بيانات أساسية
    $site_title   = isset( $_POST['sity_site_title'] ) ? sanitize_text_field( $_POST['sity_site_title'] ) : '';
    $site_tagline = isset( $_POST['sity_site_tagline'] ) ? sanitize_text_field( $_POST['sity_site_tagline'] ) : '';

    // SEO
    $seo_focus = isset( $_POST['sity_seo_focus_keyword'] ) ? sanitize_text_field( $_POST['sity_seo_focus_keyword'] ) : '';
    $seo_meta  = isset( $_POST['sity_seo_meta_description'] ) ? sanitize_textarea_field( $_POST['sity_seo_meta_description'] ) : '';

    // إعدادات إضافية
    $site_type      = isset( $_POST['sity_site_type'] ) ? sanitize_text_field( $_POST['sity_site_type'] ) : '';
    $brand_color    = isset( $_POST['sity_brand_color'] ) ? sanitize_hex_color( $_POST['sity_brand_color'] ) : '';
    $allow_indexing = isset( $_POST['sity_allow_indexing'] ) ? ( $_POST['sity_allow_indexing'] === 'yes' ? 1 : 0 ) : 1;
    $create_pages   = isset( $_POST['sity_create_basic_pages'] ) ? 1 : 0;

    // اسم الموقع والوصف
    if ( $site_title ) {
        update_option( 'blogname', $site_title );
    }
    if ( $site_tagline ) {
        update_option( 'blogdescription', $site_tagline );
    }

    // SEO
    update_option( 'sity_seo_focus_keyword', $seo_focus );
    update_option( 'sity_seo_meta_description', $seo_meta );

    // نوع الموقع + لون البراند
    update_option( 'sity_site_type', $site_type );
    if ( $brand_color ) {
        update_option( 'sity_brand_color', $brand_color );
    }

    // إعداد فهرسة محركات البحث
    // blog_public = 1 يسمح، 0 يمنع
    update_option( 'blog_public', $allow_indexing ? 1 : 0 );

    // إنشاء صفحات أساسية
    if ( $create_pages ) {
        sity_onb_maybe_create_basic_pages();
        update_option( 'sity_onb_created_basic_pages', 1 );
    }

    // رفع اللوجو لو موجود
    if ( ! empty( $_FILES['sity_site_logo']['name'] ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        $attachment_id = media_handle_upload( 'sity_site_logo', 0 );
        if ( ! is_wp_error( $attachment_id ) ) {
            set_theme_mod( 'custom_logo', $attachment_id );
        }
    }

    // ضبط الروابط الدائمة على /%postname%/
    update_option( 'permalink_structure', '/%postname%/' );
    flush_rewrite_rules();

    // علّم إن الأونبوردنج خلصت
    update_option( 'sity_onb_done', 1 );

    wp_safe_redirect( admin_url( 'index.php?onboarding=done' ) );
    exit;
}
add_action( 'admin_init', 'sity_onb_handle_post' );

/**
 * صفحة الأونبوردنج – HTML + CSS + JS
 */
function sity_onb_render_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'غير مسموح' );
    }

    $current_title       = get_option( 'blogname' );
    $current_tagline     = get_option( 'blogdescription' );
    $seo_focus           = get_option( 'sity_seo_focus_keyword', '' );
    $seo_meta            = get_option( 'sity_seo_meta_description', '' );
    $site_type           = get_option( 'sity_site_type', '' );
    $brand_color         = get_option( 'sity_brand_color', '#7c3aed' );
    $blog_public         = (int) get_option( 'blog_public', 1 );
    $created_basic_pages = (int) get_option( 'sity_onb_created_basic_pages', 0 );
    ?>
    <style>
        /* كل حاجة جوه الروت ده عشان ما تتعارضش مع ستايل ووردبريس */
        .sity-onb-root,
        .sity-onb-root * {
            box-sizing: border-box;
            font-family: system-ui, -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif !important;
        }

        .sity-onb-root {
            direction: rtl;
            min-height: 100vh;
            margin: -32px -20px -20px -20px;
            padding: 32px 16px 40px;
            background:
                radial-gradient(circle at top left, rgba(196,181,253,0.9) 0, transparent 55%),
                radial-gradient(circle at bottom right, rgba(191,219,254,0.9) 0, #f4f4ff 55%);
            display: flex;
            align-items: center;
            justify-content: center;
            overflow: auto;
        }

        .sity-onb-shell {
            max-width: 1080px;
            width: 100%;
            border-radius: 28px;
            background: linear-gradient(135deg, rgba(255,255,255,0.96), rgba(248,250,252,0.96));
            box-shadow:
                0 24px 70px rgba(148,163,184,0.45),
                0 0 0 1px rgba(199,210,254,0.9);
            position: relative;
            overflow: hidden;
            padding: 26px 26px 22px;
        }

        .sity-onb-shell::before {
            content: '';
            position: absolute;
            inset: -40%;
            background:
                radial-gradient(circle at 0 0, rgba(244,114,182,0.17), transparent 60%),
                radial-gradient(circle at 100% 100%, rgba(129,140,248,0.18), transparent 60%);
            opacity: 0.85;
            mix-blend-mode: screen;
            pointer-events: none;
        }

        .sity-onb-inner {
            position: relative;
            z-index: 2;
        }

        .sity-onb-header {
            display: flex;
            flex-wrap: wrap;
            justify-content: space-between;
            gap: 16px;
            margin-bottom: 22px;
        }

        .sity-onb-chip {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 4px 14px;
            border-radius: 999px;
            border: 1px solid rgba(199,210,254,0.9);
            background: rgba(255,255,255,0.96);
            font-size: 11px;
            letter-spacing: .12em;
            text-transform: uppercase;
            color: #4b5563;
        }

        .sity-onb-chip-dot {
            width: 8px;
            height: 8px;
            border-radius: 999px;
            background: #22c55e;
            box-shadow: 0 0 0 4px rgba(34,197,94,0.25);
        }

        .sity-onb-title {
            font-size: 26px;
            font-weight: 700;
            margin: 10px 0 4px;
            background: linear-gradient(90deg, #4c1d95, #7c3aed, #0ea5e9);
            -webkit-background-clip: text;
            background-clip: text;
            color: transparent;
        }

        .sity-onb-subtitle {
            font-size: 13px;
            color: #6b7280;
            max-width: 440px;
        }

        .sity-onb-step-info {
            text-align: left;
            font-size: 12px;
            color: #4b5563;
        }

        .sity-onb-step-info small {
            display: block;
            font-size: 11px;
            color: #9ca3af;
        }

        .sity-onb-grid {
            display: grid;
            grid-template-columns: repeat(2, minmax(0, 1fr));
            gap: 18px 24px;
            margin-bottom: 18px;
        }

        .sity-onb-section-title {
            font-size: 14px;
            font-weight: 600;
            color: #111827;
            margin-bottom: 4px;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .sity-onb-section-title span.icon {
            width: 20px;
            height: 20px;
            border-radius: 999px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            font-size: 12px;
            background: rgba(196,181,253,0.6);
            color: #4c1d95;
        }

        .sity-onb-section-hint {
            font-size: 12px;
            color: #6b7280;
            margin-bottom: 10px;
        }

        .sity-onb-field-group {
            margin-bottom: 10px;
        }

        .sity-onb-label {
            display: block;
            font-size: 13px;
            color: #374151;
            margin-bottom: 4px;
        }

        .sity-onb-input,
        .sity-onb-textarea,
        .sity-onb-select,
        .sity-onb-file {
            width: 100%;
            border-radius: 14px;
            border: 1px solid rgba(209,213,219,0.95);
            background: rgba(255,255,255,0.9);
            color: #111827;
            font-size: 13px;
            padding: 9px 11px;
        }

        .sity-onb-textarea {
            min-height: 90px;
            resize: vertical;
        }

        .sity-onb-input:focus,
        .sity-onb-textarea:focus,
        .sity-onb-select:focus {
            outline: none;
            border-color: #7c3aed;
            box-shadow: 0 0 0 1px rgba(124,58,237,0.6);
            background: #ffffff;
        }

        .sity-onb-hint {
            font-size: 11px;
            color: #9ca3af;
            margin-top: 4px;
        }

        .sity-onb-pill {
            display: inline-flex;
            align-items: center;
            gap: 6px;
            padding: 3px 10px;
            border-radius: 999px;
            background: rgba(238,242,255,0.9);
            color: #4b5563;
            font-size: 11px;
        }

        .sity-onb-toggle-group {
            display: flex;
            gap: 10px;
            align-items: center;
            flex-wrap: wrap;
        }

        .sity-onb-radio-pill {
            display: inline-flex;
            align-items: center;
            gap: 6px;
            padding: 6px 11px;
            border-radius: 999px;
            border: 1px solid rgba(209,213,219,0.9);
            background: rgba(255,255,255,0.9);
            font-size: 12px;
            cursor: pointer;
        }

        .sity-onb-radio-pill input {
            margin: 0;
        }

        .sity-onb-radio-pill.active {
            border-color: #7c3aed;
            background: rgba(237,233,254,0.9);
            color: #4c1d95;
        }

        .sity-onb-permalink-box {
            border-radius: 18px;
            border: 1px solid rgba(209,213,219,0.95);
            background: rgba(255,255,255,0.9);
            padding: 12px 14px;
            font-size: 12px;
            color: #111827;
        }

        .sity-onb-badge-success {
            padding: 3px 9px;
            border-radius: 999px;
            background: #22c55e;
            color: #f9fafb;
            font-size: 11px;
        }

        .sity-onb-permalink-box code {
            font-size: 11px;
            background: #f1f5f9;
            padding: 3px 8px;
            border-radius: 999px;
            border: 1px solid #e5e7eb;
            color: #111827;
        }

        .sity-onb-divider {
            border: none;
            border-top: 1px solid rgba(209,213,219,0.9);
            margin: 14px 0 12px;
        }

        .sity-onb-footer {
            display: flex;
            flex-wrap: wrap;
            justify-content: space-between;
            gap: 10px;
            align-items: center;
        }

        .sity-onb-footer-text {
            font-size: 11px;
            color: #6b7280;
        }

        .sity-onb-actions {
            display: flex;
            gap: 8px;
        }

        .sity-onb-btn-primary,
        .sity-onb-btn-ghost {
            border-radius: 999px;
            font-size: 13px;
            cursor: pointer;
            transition: all .18s ease;
            border: none;
        }

        .sity-onb-btn-primary {
            padding: 9px 22px;
            font-weight: 600;
            background: linear-gradient(90deg, #7c3aed, #6366f1);
            color: #f9fafb;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            box-shadow: 0 10px 26px rgba(129,140,248,0.6);
        }

        .sity-onb-btn-primary:hover {
            transform: translateY(-1px);
            filter: brightness(1.04);
        }

        .sity-onb-btn-ghost {
            padding: 9px 18px;
            border: 1px solid rgba(209,213,219,0.95);
            background: rgba(255,255,255,0.9);
            color: #374151;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            justify-content: center;
        }

        .sity-onb-btn-ghost:hover {
            background: #f3f4ff;
        }

        .sity-onb-meta-counter {
            font-size: 11px;
            color: #9ca3af;
            text-align: left;
        }

        .sity-onb-meta-counter.danger {
            color: #ef4444;
        }

        .sity-onb-inline-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 8px;
        }

        @media (max-width: 900px) {
            .sity-onb-root {
                margin: -20px -10px -10px -10px;
                padding: 16px 10px 24px;
            }
            .sity-onb-shell {
                padding: 18px 16px 16px;
            }
            .sity-onb-title {
                font-size: 22px;
            }
            .sity-onb-grid {
                grid-template-columns: 1fr;
            }
            .sity-onb-step-info {
                text-align: right;
            }
        }
    </style>

    <div class="sity-onb-root">
        <div class="sity-onb-shell">
            <div class="sity-onb-inner">
                <div class="sity-onb-header">
                    <div>
                        <div class="sity-onb-chip">
                            <span class="sity-onb-chip-dot"></span>
                            <span>Sity Cloud · إعداد أولي</span>
                        </div>
                        <h1 class="sity-onb-title">جهّز هوية ونواة موقعك في دقايق</h1>
                        <p class="sity-onb-subtitle">
                            من هنا بتعرف منصتك على مشروعك: اسم، وصف، نوع الموقع، ألوان البراند، SEO أساسي، صفحات مهمة، ولوجو. بعد الخطوة دي، تقدر تركّز على المحتوى.
                        </p>
                    </div>
                    <div class="sity-onb-step-info">
                        الخطوة 1 من 1<br>
                        <small>Onboarding سريع · يُفضّل عدم تخطيه</small>
                    </div>
                </div>

                <form method="post" enctype="multipart/form-data">
                    <?php wp_nonce_field( 'sity_onb_form', 'sity_onb_nonce' ); ?>

                    <div class="sity-onb-grid">
                        <!-- هوية الموقع -->
                        <div>
                            <div class="sity-onb-section-title">
                                <span class="icon">🧩</span>
                                هوية الموقع
                            </div>
                            <div class="sity-onb-section-hint">
                                الاسم والوصف بيظهروا للزوار وفي نتائج البحث، فخليهم واضحين ومناسبين لهوية البراند.
                            </div>

                            <div class="sity-onb-field-group">
                                <label for="sity_site_title" class="sity-onb-label">اسم الموقع</label>
                                <input type="text"
                                       id="sity_site_title"
                                       name="sity_site_title"
                                       class="sity-onb-input"
                                       value="<?php echo esc_attr( $current_title ); ?>"
                                       placeholder="مثال: منصة Sity Cloud لإنشاء المواقع"
                                       required>
                            </div>

                            <div class="sity-onb-field-group">
                                <label for="sity_site_tagline" class="sity-onb-label">وصف قصير (Tagline)</label>
                                <input type="text"
                                       id="sity_site_tagline"
                                       name="sity_site_tagline"
                                       class="sity-onb-input"
                                       value="<?php echo esc_attr( $current_tagline ); ?>"
                                       placeholder="مثال: أنشئ موقعك الاحترافي في دقائق بدون خبرة تقنية">
                            </div>

                            <div class="sity-onb-field-group">
                                <label for="sity_site_type" class="sity-onb-label">نوع الموقع</label>
                                <select id="sity_site_type"
                                        name="sity_site_type"
                                        class="sity-onb-select">
                                    <?php
                                    $types = array(
                                        ''           => 'اختار نوع الموقع…',
                                        'business'   => 'موقع شركة / بيزنس',
                                        'store'      => 'متجر إلكتروني',
                                        'portfolio'  => 'بورتفوليو / أعمال',
                                        'blog'       => 'مدونة شخصية / محتوى',
                                        'restaurant' => 'مطعم / كافيه',
                                        'service'    => 'خدمة / حجز مواعيد',
                                        'landing'    => 'صفحة هبوط لحملة معينة',
                                    );
                                    foreach ( $types as $value => $label ) {
                                        printf(
                                            '<option value="%s"%s>%s</option>',
                                            esc_attr( $value ),
                                            selected( $site_type, $value, false ),
                                            esc_html( $label )
                                        );
                                    }
                                    ?>
                                </select>
                                <div class="sity-onb-hint">
                                    بنستخدم نوع الموقع في تحسين الإعدادات الافتراضية واقتراحات التصميم لاحقًا.
                                </div>
                            </div>
                        </div>

                        <!-- SEO الأساسي -->
                        <div>
                            <div class="sity-onb-section-title">
                                <span class="icon">🔍</span>
                                إعدادات SEO الأساسية
                            </div>
                            <div class="sity-onb-section-hint">
                                إعداد بسيط يساعدك تبدأ صح في جوجل. تقدر تضيف بلجن SEO بعد كده وتوسع الإعدادات براحتك.
                            </div>

                            <div class="sity-onb-field-group">
                                <label for="sity_seo_focus_keyword" class="sity-onb-label">الكلمة المفتاحية الرئيسية</label>
                                <input type="text"
                                       id="sity_seo_focus_keyword"
                                       name="sity_seo_focus_keyword"
                                       class="sity-onb-input"
                                       value="<?php echo esc_attr( $seo_focus ); ?>"
                                       placeholder="مثال: إنشاء موقع إلكتروني بدون مبرمج">
                                <div class="sity-onb-hint">
                                    كلمة أو جملة متوقع إن العميل يكتبها في جوجل علشان يوصل لموقعك.
                                </div>
                            </div>

                            <div class="sity-onb-field-group">
                                <label for="sity_seo_meta_description" class="sity-onb-label">وصف ميتا للصفحة الرئيسية</label>
                                <textarea
                                    id="sity_seo_meta_description"
                                    name="sity_seo_meta_description"
                                    class="sity-onb-textarea"
                                    placeholder="وصف مختصر يظهر في نتائج البحث… الأفضل بين 120 و 160 حرف."><?php echo esc_textarea( $seo_meta ); ?></textarea>
                            </div>

                            <div class="sity-onb-inline-row">
                                <div id="sity_onb_meta_counter" class="sity-onb-meta-counter"></div>
                                <div class="sity-onb-hint">
                                    مثال: “منصة Sity Cloud بتخليك تبني موقعك بنفسك في دقائق بدون خبرة تقنية.”
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="sity-onb-grid" style="margin-top:10px;">
                        <!-- لون البراند + فهرسة -->
                        <div>
                            <div class="sity-onb-section-title">
                                <span class="icon">🎨</span>
                                ألوان البراند ومحركات البحث
                            </div>
                            <div class="sity-onb-section-hint">
                                اختار اللون الرئيسي لهويتك، وحدد إذا كنت حابب تسمح لمحركات البحث إنها تفهرس موقعك من دلوقتي ولا تستنى.
                            </div>

                            <div class="sity-onb-field-group">
                                <label for="sity_brand_color" class="sity-onb-label">اللون الرئيسي للبراند</label>
                                <div class="sity-onb-inline-row">
                                    <input type="color"
                                           id="sity_brand_color"
                                           name="sity_brand_color"
                                           class="sity-onb-input"
                                           style="max-width:70px;padding:2px 4px;"
                                           value="<?php echo esc_attr( $brand_color ); ?>">
                                    <div class="sity-onb-hint">
                                        هنستخدم اللون ده لاحقًا في الثيم والبلوكات الجاهزة.
                                    </div>
                                </div>
                            </div>

                            <div class="sity-onb-field-group">
                                <span class="sity-onb-label">فهرسة محركات البحث (Search Engines)</span>
                                <div class="sity-onb-toggle-group">
                                    <label class="sity-onb-radio-pill <?php echo $blog_public ? 'active' : ''; ?>">
                                        <input type="radio"
                                               name="sity_allow_indexing"
                                               value="yes"
                                               <?php checked( $blog_public, 1 ); ?>>
                                        <span>السماح بالفهرسة</span>
                                    </label>
                                    <label class="sity-onb-radio-pill <?php echo ! $blog_public ? 'active' : ''; ?>">
                                        <input type="radio"
                                               name="sity_allow_indexing"
                                               value="no"
                                               <?php checked( $blog_public, 0 ); ?>>
                                        <span>إيقاف الفهرسة مؤقتًا</span>
                                    </label>
                                </div>
                                <div class="sity-onb-hint">
                                    لو لسه بتجهز المحتوى أو الموقع في مرحلة تجارب، ممكن توقف الفهرسة مؤقتًا.
                                </div>
                            </div>
                        </div>

                        <!-- لوجو + الروابط الدائمة + صفحات أساسية -->
                        <div>
                            <div class="sity-onb-section-title">
                                <span class="icon">✨</span>
                                الشعار والروابط وصفحات أساسية
                            </div>
                            <div class="sity-onb-section-hint">
                                حمّل لوجو بسيط، وخلي الروابط مرتّبة، وأنشئ الصفحات الأساسية بضغطة واحدة.
                            </div>

                            <div class="sity-onb-field-group">
                                <label for="sity_site_logo" class="sity-onb-label">رفع لوجو (Logo)</label>
                                <input type="file"
                                       id="sity_site_logo"
                                       name="sity_site_logo"
                                       class="sity-onb-file"
                                       accept="image/*">
                                <div class="sity-onb-hint">
                                    يفضّل PNG أو SVG بخلفية شفافة ومقاس أفقي تقريبًا (مثلًا 200×60).
                                </div>
                            </div>

                            <div class="sity-onb-field-group">
                                <span class="sity-onb-label">الروابط الدائمة (Permalinks)</span>
                                <div class="sity-onb-permalink-box">
                                    <div class="sity-onb-inline-row" style="margin-bottom:4px;">
                                        <span>هيتم تفعيل صيغة <strong>Post Name</strong> تلقائيًا</span>
                                        <span class="sity-onb-badge-success">مُفعّل</span>
                                    </div>
                                    <div>
                                        <code>https://yourdomain.com/صفحتك/</code>
                                    </div>
                                    <div class="sity-onb-hint" style="margin-top:6px;">
                                        ده الشكل الأنظف والمفضل لمحركات البحث بدل الروابط العشوائية.
                                    </div>
                                </div>
                            </div>

                            <div class="sity-onb-field-group" style="margin-top:8px;">
                                <label class="sity-onb-label">الصفحات الأساسية</label>
                                <label style="display:flex;align-items:center;gap:6px;font-size:12px;color:#374151;cursor:pointer;">
                                    <input type="checkbox"
                                           name="sity_create_basic_pages"
                                           value="1"
                                           <?php checked( $created_basic_pages, 1 ); ?>>
                                    <span>إنشاء صفحات أساسية تلقائيًا: من نحن، تواصل معنا، سياسة الخصوصية</span>
                                </label>
                                <div class="sity-onb-hint">
                                    لو الصفحات موجودة بالفعل مش هنكررها، هنضيف بس الناقص.
                                </div>
                            </div>
                        </div>
                    </div>

                    <hr class="sity-onb-divider">

                    <div class="sity-onb-footer">
                        <div class="sity-onb-footer-text">
                            بعد الحفظ، هنعيد توجيهك للوحة التحكم. تقدر ترجع تعدّل كل الإعدادات دي من إعدادات الموقع والثيم في أي وقت.
                        </div>
                        <div class="sity-onb-actions">
                            <a href="<?php echo esc_url( admin_url() ); ?>" class="sity-onb-btn-ghost">
                                تخطي الآن والرجوع للوحة التحكم
                            </a>
                            <button type="submit" name="sity_onb_submit" class="sity-onb-btn-primary">
                                حفظ وإنهاء الإعداد <span>⚡</span>
                            </button>
                        </div>
                    </div>
                </form>

            </div>
        </div>
    </div>

    <script>
        (function() {
            // عداد وصف الميتا
            var textarea = document.getElementById('sity_seo_meta_description');
            var counter  = document.getElementById('sity_onb_meta_counter');

            function updateCounter() {
                if (!textarea || !counter) return;
                var len = textarea.value.length;
                counter.textContent = len + ' / 160 حرف';
                if (len > 160) {
                    counter.classList.add('danger');
                } else {
                    counter.classList.remove('danger');
                }
            }

            if (textarea) {
                textarea.addEventListener('input', updateCounter);
                updateCounter();
            }

            // تفعيل شكل راديو الفهرسة
            var pills = document.querySelectorAll('.sity-onb-radio-pill');
            pills.forEach(function(pill) {
                pill.addEventListener('click', function() {
                    pills.forEach(function(p) { p.classList.remove('active'); });
                    pill.classList.add('active');
                    var input = pill.querySelector('input[type=\"radio\"]');
                    if (input) {
                        input.checked = true;
                    }
                });
            });
        })();
    </script>
    <?php
}
