<?php
/**
 * Plugin Name: Sity Cloud Dashboard Tutorials
 * Description: استبدال محتوى wp-admin/index.php في المواقع الفرعية بصفحة فيديوهات تعليمية لمنصة Sity Cloud بدون أي إشعارات أو ودجات.
 * Version: 1.2
 * Author: Sity Cloud
 * Network: true
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * هل نحن الآن في Dashboard موقع فرعي؟
 */
function sitycloud_is_subsite_dashboard() {
    // خارج واجهة الأدمن
    if ( ! is_admin() ) {
        return false;
    }

    // Network أو User admin؟ تجاهل
    if ( is_network_admin() || is_user_admin() ) {
        return false;
    }

    // لازم نكون في index.php
    global $pagenow;
    if ( $pagenow !== 'index.php' ) {
        return false;
    }

    // لو Multisite والموقع الرئيسي → سيبه عادي
    if ( is_multisite() && is_main_site() ) {
        return false;
    }

    return true;
}

/**
 * جلب قائمة الفيديوهات من إعدادات الشبكة (أو من الموقع لو مش Multisite)
 */
function sitycloud_get_dashboard_videos() {
    if ( is_multisite() ) {
        $videos = get_site_option( 'sitycloud_dashboard_videos', array() );
    } else {
        $videos = get_option( 'sitycloud_dashboard_videos', array() );
    }

    if ( ! is_array( $videos ) ) {
        $videos = array();
    }

    return $videos;
}

/**
 * حفظ قائمة الفيديوهات
 */
function sitycloud_save_dashboard_videos( $videos ) {
    if ( ! is_array( $videos ) ) {
        $videos = array();
    }

    $clean = array();
    $i     = 0;

    foreach ( $videos as $video ) {
        // تجاهل السطور الفاضية
        $title = isset( $video['title'] ) ? trim( $video['title'] ) : '';
        $url   = isset( $video['url'] ) ? trim( $video['url'] ) : '';

        if ( $title === '' && $url === '' ) {
            continue;
        }

        $clean[ $i ] = array(
            'title'       => sanitize_text_field( $title ),
            'url'         => esc_url_raw( $url ),
            'description' => isset( $video['description'] ) ? sanitize_textarea_field( $video['description'] ) : '',
            'level'       => isset( $video['level'] ) ? sanitize_text_field( $video['level'] ) : '',
        );
        $i++;
    }

    if ( is_multisite() ) {
        update_site_option( 'sitycloud_dashboard_videos', $clean );
    } else {
        update_option( 'sitycloud_dashboard_videos', $clean );
    }
}

/**
 * إضافة صفحة إعدادات في Network Admin لإدارة الفيديوهات
 */
function sitycloud_add_network_admin_menu() {
    if ( ! is_multisite() ) {
        return;
    }

    add_menu_page(
        'Sity Cloud Tutorials',
        'Sity Cloud Tutorials',
        'manage_network_options',
        'sitycloud-dashboard-tutorials',
        'sitycloud_render_network_settings_page',
        'dashicons-welcome-learn-more',
        60
    );
}
add_action( 'network_admin_menu', 'sitycloud_add_network_admin_menu' );

/**
 * صفحة إعدادات إدارة الفيديوهات في الشبكة
 */
function sitycloud_render_network_settings_page() {
    if ( ! current_user_can( 'manage_network_options' ) ) {
        wp_die( 'ليس لديك صلاحية كافية لعرض هذه الصفحة.' );
    }

    // حفظ البيانات
    if ( isset( $_POST['sitycloud_save_videos'] ) ) {
        check_admin_referer( 'sitycloud_save_videos_nonce', 'sitycloud_save_videos_nonce_field' );
        $videos = isset( $_POST['videos'] ) ? (array) $_POST['videos'] : array();
        sitycloud_save_dashboard_videos( $videos );
        echo '<div class="updated notice"><p>تم حفظ الفيديوهات بنجاح.</p></div>';
    }

    $videos = sitycloud_get_dashboard_videos();

    // نسمح مثلاً لحد 10 فيديوهات ثابتة في الفورم
    $max_rows = 10;
    ?>
    <div class="wrap">
        <h1>فيديوهات لوحة التحكم - Sity Cloud</h1>
        <p>من هنا تقدر تضيف فيديوهات تعليمية هتظهر في صفحة Dashboard للمواقع الفرعية بدل الداشبورد الافتراضي.</p>

        <form method="post">
            <?php wp_nonce_field( 'sitycloud_save_videos_nonce', 'sitycloud_save_videos_nonce_field' ); ?>

            <table class="widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 20%;">عنوان الفيديو</th>
                        <th style="width: 30%;">رابط الفيديو (YouTube / Vimeo / إلخ)</th>
                        <th style="width: 15%;">مستوى (اختياري)</th>
                        <th style="width: 30%;">وصف قصير</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    for ( $row = 0; $row < $max_rows; $row++ ) :
                        $title       = isset( $videos[ $row ]['title'] ) ? $videos[ $row ]['title'] : '';
                        $url         = isset( $videos[ $row ]['url'] ) ? $videos[ $row ]['url'] : '';
                        $level       = isset( $videos[ $row ]['level'] ) ? $videos[ $row ]['level'] : '';
                        $description = isset( $videos[ $row ]['description'] ) ? $videos[ $row ]['description'] : '';
                    ?>
                        <tr>
                            <td>
                                <input type="text" name="videos[<?php echo esc_attr( $row ); ?>][title]" value="<?php echo esc_attr( $title ); ?>" class="regular-text" />
                            </td>
                            <td>
                                <input type="text" name="videos[<?php echo esc_attr( $row ); ?>][url]" value="<?php echo esc_attr( $url ); ?>" class="regular-text" />
                            </td>
                            <td>
                                <input type="text" name="videos[<?php echo esc_attr( $row ); ?>][level]" value="<?php echo esc_attr( $level ); ?>" class="regular-text" />
                            </td>
                            <td>
                                <textarea name="videos[<?php echo esc_attr( $row ); ?>][description]" rows="2" class="large-text"><?php echo esc_textarea( $description ); ?></textarea>
                            </td>
                        </tr>
                    <?php endfor; ?>
                </tbody>
            </table>

            <p class="description">
                اترك السطور الفارغة بدون بيانات، وسيتم تجاهلها تلقائيًا.
            </p>

            <p class="submit">
                <button type="submit" name="sitycloud_save_videos" class="button button-primary">حفظ الفيديوهات</button>
            </p>
        </form>
    </div>
    <?php
}

/**
 * إلغاء كل ودجات الداشبورد الافتراضية واستبدالها بواحدة بتاعتنا
 */
function sitycloud_customize_dashboard_widgets() {
    if ( ! sitycloud_is_subsite_dashboard() ) {
        return;
    }

    global $wp_meta_boxes;

    // مسح كل الودجات في الداشبورد
    if ( isset( $wp_meta_boxes['dashboard'] ) && is_array( $wp_meta_boxes['dashboard'] ) ) {
        $wp_meta_boxes['dashboard'] = array(
            'normal' => array(),
            'side'   => array(),
        );
    }

    // إضافة ودجت واحدة فقط فيها محتوى Sity Cloud
    add_meta_box(
        'sitycloud_dashboard_tutorials',
        'لوحة تعليم منصة Sity Cloud',
        'sitycloud_render_dashboard_tutorials_box',
        'dashboard',
        'normal',
        'high'
    );

    // إلغاء لوحة الترحيب الافتراضية
    remove_action( 'welcome_panel', 'wp_welcome_panel' );
}
add_action( 'wp_dashboard_setup', 'sitycloud_customize_dashboard_widgets', 999 );

/**
 * إخفاء كل الإشعارات في صفحة index.php للمواقع الفرعية
 */
function sitycloud_remove_dashboard_notices() {
    if ( ! sitycloud_is_subsite_dashboard() ) {
        return;
    }

    global $wp_filter;

    $hooks = array(
        'admin_notices',
        'all_admin_notices',
        'network_admin_notices',
        'user_admin_notices',
    );

    foreach ( $hooks as $hook ) {
        if ( isset( $wp_filter[ $hook ] ) ) {
            // في الإصدارات الحديثة من ووردبريس الكائن بيبقى WP_Hook
            if ( is_object( $wp_filter[ $hook ] ) && isset( $wp_filter[ $hook ]->callbacks ) ) {
                $wp_filter[ $hook ]->callbacks = array();
            } else {
                // في حالات نادرة يكون مصفوفة مباشرة
                $wp_filter[ $hook ] = array();
            }
        }
    }
}
add_action( 'in_admin_header', 'sitycloud_remove_dashboard_notices', 0 );

/**
 * محتوى ودجت Sity Cloud في صفحة Dashboard
 */
function sitycloud_render_dashboard_tutorials_box() {
    $videos = sitycloud_get_dashboard_videos();
    ?>
    <style>
        .sitycloud-dashboard-wrapper {
            margin: -12px -12px -12px -12px;
            padding: 24px;
            background: #f6f7fb;
        }
        .sitycloud-dashboard-header {
            margin-bottom: 20px;
        }
        .sitycloud-dashboard-header h2 {
            margin: 0 0 8px 0;
            font-size: 20px;
        }
        .sitycloud-dashboard-header p {
            margin: 0;
            color: #555d66;
        }
        .sitycloud-videos-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(260px, 1fr));
            gap: 16px;
            margin-top: 20px;
        }
        .sitycloud-video-card {
            background: #ffffff;
            border-radius: 8px;
            padding: 16px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.06);
            border: 1px solid #e2e4e7;
        }
        .sitycloud-video-title {
            font-size: 15px;
            font-weight: 600;
            margin: 0 0 6px 0;
        }
        .sitycloud-video-level {
            display: inline-block;
            font-size: 11px;
            padding: 2px 6px;
            border-radius: 999px;
            background: #f1f5ff;
            color: #3858e9;
            margin-bottom: 6px;
        }
        .sitycloud-video-desc {
            font-size: 13px;
            color: #555d66;
            margin-bottom: 10px;
        }
        .sitycloud-video-link {
            display: inline-block;
            font-size: 13px;
            text-decoration: none;
            padding: 6px 12px;
            border-radius: 4px;
            background: #3858e9;
            color: #ffffff;
        }
        .sitycloud-video-link:hover {
            opacity: 0.9;
        }
        .sitycloud-no-videos {
            font-size: 14px;
            color: #555d66;
            margin-top: 12px;
        }
    </style>

    <div class="sitycloud-dashboard-wrapper">
        <div class="sitycloud-dashboard-header">
            <h2>مرحبًا بك في لوحة تحكم موقعك على منصة Sity Cloud</h2>
            <p>هنا تقدر تشوف فيديوهات تعليمية تساعدك في استخدام المنصة وإنشاء موقعك خطوة بخطوة.</p>
        </div>

        <?php if ( empty( $videos ) ) : ?>
            <p class="sitycloud-no-videos">
                لم يتم إضافة فيديوهات تعليمية بعد. من فضلك تواصل مع إدارة المنصة لإضافة الفيديوهات من لوحة تحكم الشبكة.
            </p>
        <?php else : ?>
            <div class="sitycloud-videos-grid">
                <?php foreach ( $videos as $video ) : ?>
                    <div class="sitycloud-video-card">
                        <?php if ( ! empty( $video['level'] ) ) : ?>
                            <div class="sitycloud-video-level">
                                <?php echo esc_html( $video['level'] ); ?>
                            </div>
                        <?php endif; ?>

                        <?php if ( ! empty( $video['title'] ) ) : ?>
                            <div class="sitycloud-video-title">
                                <?php echo esc_html( $video['title'] ); ?>
                            </div>
                        <?php endif; ?>

                        <?php if ( ! empty( $video['description'] ) ) : ?>
                            <div class="sitycloud-video-desc">
                                <?php echo esc_html( $video['description'] ); ?>
                            </div>
                        <?php endif; ?>

                        <?php if ( ! empty( $video['url'] ) ) : ?>
                            <a href="<?php echo esc_url( $video['url'] ); ?>" target="_blank" class="sitycloud-video-link">
                                مشاهدة الفيديو
                            </a>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
    <?php
}
