<?php
/**
 * This file is reponsible for initializing ads tracking.
 *
 * @package wp-analytify-pro
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Analytify_Ads_WooCommerce
 *
 * This class handles Google Ads conversion tracking integration with WooCommerce.
 */
class Analytify_Ads_WooCommerce {

	/**
	 * @var string The ID of the Google Ads conversion.
	 */
	public $ads_conversion_id = '';

	/**
	 * @var string The label of the Google Ads conversion.
	 */
	public $ads_conversion_label = '';

	/**
	 * @var Analytify_WooCommerce_Conversions_Helpers The instance of the helper trait.
	 */
	use Analytify_WooCommerce_Conversions_Helpers;

	/**
	 * Analytify_Ads_WooCommerce constructor.
	 *
	 * @param string $ads_conversion_id    The ID of the Google Ads conversion.
	 * @param string $ads_conversion_label The label of the Google Ads conversion.
	 */
	public function __construct( string $ads_conversion_id, string $ads_conversion_label ) {
		$this->ads_conversion_id    = $ads_conversion_id;
		$this->ads_conversion_label = $ads_conversion_label;

		$this->hooks();
	}

	/**
	 * Add conversion related hooks.
	 */
	public function hooks() {
		add_action( 'analytify_tracking_code_after_pageview', array( $this, 'gtag_conversion_call' ) );
	}

	/**
	 * Call the Google Ads conversion when conditions are met.
	 */
	public function gtag_conversion_call() {
		if ( apply_filters( 'analytify_ads_is_conversion_page', $this->conversion_checks() ) ) {
            echo $this->conversion_tracking_code(); // phpcs:ignore
			$this->set_order_tracked();
		}
	}

	/**
	 * Generate the conversion tracking code.
	 *
	 * @return string The generated conversion tracking code.
	 */
	public function conversion_tracking_code() {
		$code = '';

		if ( ! $this->is_tracked() ) {
			$code = stripslashes(
				"\ngtag( 'event', 'conversion', " . wp_json_encode( $this->get_conversion_info() ) . " );"
			);
		}

		return $code;
	}

	/**
	 * Get the conversion event information.
	 *
	 * @return iterable An array containing conversion event information.
	 */
	public function get_conversion_info() {
		$conversion_event_info = array(
			'send_to' => $this->ads_conversion_id . '/' . $this->ads_conversion_label,
		);

		if ( $this->get_order_total() ) {
			$conversion_event_info['value'] = $this->get_order_total();
		}

		if ( $this->get_order_number() ) {
			$conversion_event_info['transaction_id'] = $this->get_order_number();
		}

		if ( $this->get_order_currency() ) {
			$conversion_event_info['currency'] = $this->get_order_currency();
		}

		return $conversion_event_info;
	}
}
