<?php

/**
 * This trait provides helper methods related to WooCommerce conversions tracking.
 */
trait Analytify_WooCommerce_Conversions_Helpers {

	/**
	 * @var int|null The WooCommerce order ID associated with the conversion.
	 */
	public $wc_order_id;

	/**
	 * @var WC_Order|null The WooCommerce order object associated with the conversion.
	 */
	public $wc_order;

	/**
	 * Checks if the current page is the order received page.
	 *
	 * @return bool Returns true if the current page is the order received page, false otherwise.
	 */
	public function conversion_checks() {
		if ( function_exists( 'is_order_received_page' ) && is_order_received_page() ) {
			return true;
		}
		return false;
	}

	/**
	 * Checks if the conversion is tracked for the current order.
	 *
	 * @return bool Returns true if the conversion is tracked, false otherwise.
	 */
	public function is_tracked() {
		if ( $this->get_order() ) {
			$tracked = $this->get_order()->get_meta( 'analytify_ads_conversion_tracked' );
		}
		return ! empty( $tracked ) && 'yes' === $tracked;
	}

	/**
	 * Retrieves the WooCommerce order object associated with the current order number.
	 *
	 * @return bool|WC_Order Returns the WooCommerce order object if found, otherwise false.
	 */
	public function get_order() {
		if ( ! isset( $this->order ) ) {
			if ( $this->get_order_number() ) {
				$this->order = wc_get_order( $this->get_order_number() );
			} else {
				$this->order = false;
			}
		}
		return $this->order;
	}

	/**
	 * Retrieves the WooCommerce order number from the query variables.
	 *
	 * @return bool|int Returns the WooCommerce order number if found, otherwise false.
	 */
	public function get_order_number() {
		if ( ! isset( $this->wc_order_id ) ) {
			$wc_order_id = absint( get_query_var( 'order-received' ) );
			if ( empty( $wc_order_id ) || 0 === $wc_order_id ) {
				$this->wc_order_id = false;
			} else {
				$this->wc_order_id = $wc_order_id;
			}
		}
		return $this->wc_order_id;
	}

	/**
	 * Retrieves the total amount of the current order.
	 *
	 * @return bool|float Returns the order total amount if the order is available, otherwise false.
	 */
	public function get_order_total() {
		if ( $this->get_order() ) {
			return (float) $this->get_order()->get_total();
		}
		return false;
	}

	/**
	 * Retrieves the currency of the current order.
	 *
	 * @return bool|string Returns the order currency if the order is available, otherwise false.
	 */
	public function get_order_currency() {
		if ( $this->get_order() ) {
			return $this->get_order()->get_currency();
		}
		return false;
	}

	/**
	 * Sets the conversion tracked status for the current order.
	 */
	public function set_order_tracked() {
		if ( $this->get_order() ) {
			$this->get_order()->add_meta_data( 'analytify_ads_conversion_tracked', 'yes', true );
			$this->get_order()->save_meta_data();
		}
	}
}
