<?php
/**
 * List performance view with ECharts Funnel.
 *
 * @param object $current Current object.
 * @param object $stats  Statistics data.
 * 
 * @return void
 */

function wpa_edd_funnel_chart($current, $stats)
{

    $data = [];
    $conversion_rates = [
        'View to Cart' => 0,
        'Cart to Purchase' => 0,
    ];
    $abandonment_rates = [
        'View Abandonment' => 0,
        'Cart Abandonment' => 0,
    ];
    $values = [];

    if (!empty($stats['rows'])) {
        foreach ($stats['rows'] as $stat) {
            // Safely extract values or default to 0
            $viewed = isset($stat['itemsViewed']) ? (int) $stat['itemsViewed'] : 0;
            $added = isset($stat['itemsAddedToCart']) ? (int) $stat['itemsAddedToCart'] : 0;
            $purchased = isset($stat['itemsPurchased']) ? (int) $stat['itemsPurchased'] : 0;

            // Prepare data for the funnel chart
            $data = [
                [
                    'value' => 1500,
                    'name' => 'View Item',
                    'itemStyle' => ['color' => '#f5a623'],
                    'actual' => $viewed,
                ],
                [
                    'value' => 1000,
                    'name' => 'Add to Cart',
                    'itemStyle' => ['color' => '#7ed321'],
                    'actual' => $added,
                ],
                [
                    'value' => 500,
                    'name' => 'Purchase',
                    'itemStyle' => ['color' => '#4a90e2'],
                    'actual' => $purchased,
                ],
            ];

            // Collect values for finding min and max
            $values = array_merge($values, [$viewed, $added, $purchased]);

            // Calculate conversion and abandonment rates by stages
            $conversion_rates = [
                'View to Cart' => ($viewed > 0) ? round(($added / $viewed) * 100, 2) : 0,
                'Cart to Purchase' => ($added > 0) ? round(($purchased / $added) * 100, 2) : 0,
            ];

            $abandonment_rates = [
                'View Abandonment' => ($viewed > 0) ? round((($viewed - $added) / $viewed) * 100, 2) : 0,
                'Cart Abandonment' => ($added > 0) ? round((($added - $purchased) / $added) * 100, 2) : 0,
            ];
        }
    } else {
        $data = [
            [
                'value' => 1500,
                'name' => 'View Item',
                'itemStyle' => ['color' => '#f5a623'],
                'actual' => 0,
            ],
            [
                'value' => 1000,
                'name' => 'Add to Cart',
                'itemStyle' => ['color' => '#7ed321'],
                'actual' => 0,
            ],
            [
                'value' => 500,
                'name' => 'Purchase',
                'itemStyle' => ['color' => '#4a90e2'],
                'actual' => 0,
            ],
        ];
    }

    ?>

    <div class="analytify_general_status analytify_status_box_wraper">
        <div class="analytify_status_header">
            <h3><?php esc_html_e('EDD Funnel', 'wp-analytify-edd'); ?></h3>
        </div>

        <table class="analytify_data_tables">
            <thead>
                <tr>
                    <th class="no-header"></th>
                    <th class="right-align header-center">
                        <?php esc_html_e('Conversion Rate by Stages', 'wp-analytify-edd'); ?>
                    </th>
                    <th class="right-align header-center">
                        <?php esc_html_e('Abandonment Rate', 'wp-analytify-edd'); ?>
                    </th>
                </tr>
            </thead>
        </table>

        <div class="analytify_status_body">
            <div class="funnel-wrapper">
                <div class="analytify_funnel-chart-container">
                    <div class="analytify_funnel-chart">
                        <div id="analytify_funnel-chart"></div>
                    </div>

                    <div class="conversion-rates">
                        <p class="view-conversion"><?php echo esc_html($conversion_rates['View to Cart']); ?>%</p>
                        <p class="cart-conversion"><?php echo esc_html($conversion_rates['Cart to Purchase']); ?>%</p>
                    </div>

                    <div class="abandonment-rates">
                        <p><?php echo esc_html($abandonment_rates['View Abandonment']); ?>%</p>
                        <p><?php echo esc_html($abandonment_rates['Cart Abandonment']); ?>%</p>
                    </div>
                </div>
            </div>
        </div>
    </div>




    <script type="text/javascript">
        document.addEventListener("DOMContentLoaded", function () {
            var analytify_edd_funnel_chartId = document.getElementById('analytify_funnel-chart');
            var analytify_edd_chart = echarts.init(analytify_edd_funnel_chartId);

            // Dynamic data from PHP
            var funnelData = <?php echo json_encode($data); ?>;

            var option = {
                series: [
                    {
                        name: 'Conversion Rate',
                        type: 'funnel',
                        left: '10%',
                        top: 20,
                        bottom: 20,
                        width: '80%',
                        min: 0,
                        max: 1500,
                        minSize: '0%',
                        maxSize: '100%',
                        sort: 'descending',
                        gap: 10,
                        label: {
                            show: true,
                            position: 'inside',
                            formatter: function (params) {
                                return `${params.name}\n${params.data.actual}`;
                            },
                            color: '#fff',
                            fontSize: 17
                        },
                        labelLine: {
                            show: true,
                            length: 30,
                            lineStyle: {
                                type: 'dashed',
                                color: '#aaa',
                            },
                        },
                        itemStyle: {
                            borderColor: '#fff',
                            borderWidth: 2,
                            borderRadius: 8,
                        },
                        emphasis: {
                            label: {
                                fontSize: 19,
                            },
                        },
                        data: funnelData,
                    },
                ],
            };

            analytify_edd_chart.setOption(option);
            window.addEventListener('resize', function () {
                analytify_edd_chart.resize();
            });
        });
    </script>

    <?php
}


