<?php
/**
 * Plugin Name: SityCloud Wallet & Payments
 * Description: نظام محفظة + شحن رصيد عن طريق WooCommerce + الدفع من المحفظة، مع شورت كود لعرض معاملات العميل ولوحة تحكم للأدمن.
 * Author: Sity Cloud
 * Version: 1.1.0
 * Text Domain: sitycloud-wallet
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class SityCloud_Wallet_Plugin {

    const META_BALANCE_KEY  = '_sitycloud_wallet_balance';
    const OPTION_PRODUCT_ID = 'sitycloud_wallet_topup_product_id';
    const TABLE_NAME        = 'sitycloud_wallet_tx';

    /** @var SityCloud_Wallet_Plugin */
    private static $instance = null;

    /**
     * Singleton init
     */
    public static function init() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Helper للاستخدام من الجateway
     */
    public static function get_instance() {
        return self::init();
    }

    /**
     * تهيئة الهواكس
     */
    private function __construct() {

        // شورت كود الواجهة الأمامية
        add_shortcode( 'sitycloud_wallet', array( $this, 'wallet_shortcode' ) );

        // معالجة فورم الشحن من الفرونت
        add_action( 'init', array( $this, 'handle_frontend_topup' ) );

        // دمج سعر الشحن مع كارت ووكومرس
        add_action( 'woocommerce_before_calculate_totals', array( $this, 'set_cart_item_price_from_meta' ), 20, 1 );

        // تخزين بيانات الشحن في الـ order item
        add_action( 'woocommerce_checkout_create_order_line_item', array( $this, 'save_cart_meta_to_order_item' ), 10, 4 );

        // بعد اكتمال الطلب → إضافة رصيد للمحفظة
        add_action( 'woocommerce_order_status_completed', array( $this, 'handle_order_completed_topup' ) );

        // تسجيل بوابة الدفع بمحفظة SityCloud
        add_filter( 'woocommerce_payment_gateways', array( $this, 'register_wallet_gateway' ) );

        // لوحة التحكم للأدمن
        add_action( 'admin_menu', array( $this, 'register_admin_menu' ) );
        add_action( 'admin_post_sitycloud_wallet_adjust', array( $this, 'handle_admin_adjust_balance' ) );

        // تنبيه لو WooCommerce مش متفعّل
        add_action( 'admin_notices', array( $this, 'admin_wc_notice' ) );
    }

    /**
     * أثناء التفعيل: إنشاء جدول + محاولة إنشاء منتج الشحن
     */
    public static function activate() {
        global $wpdb;

        $table_name      = $wpdb->prefix . self::TABLE_NAME;
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            user_id BIGINT(20) UNSIGNED NOT NULL,
            type VARCHAR(20) NOT NULL,
            amount DECIMAL(20,4) NOT NULL,
            description TEXT NULL,
            order_id BIGINT(20) UNSIGNED DEFAULT NULL,
            balance_after DECIMAL(20,4) NOT NULL DEFAULT 0,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY  (id),
            KEY user_id (user_id),
            KEY order_id (order_id)
        ) {$charset_collate};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta( $sql );

        // لو WooCommerce متفعّل وقت التفعيل، جرّب تنشئ منتج الشحن
        if ( class_exists( 'WooCommerce' ) ) {
            $self = self::get_instance();
            $self->maybe_create_topup_product();
        }
    }

    /**
     * تنبيه للأدمن لو WooCommerce مش شغال
     */
    public function admin_wc_notice() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        if ( ! class_exists( 'WooCommerce' ) ) {
            echo '<div class="notice notice-error"><p>بلجن <strong>SityCloud Wallet</strong> يحتاج WooCommerce ليعمل بشكل كامل (الشحن / الدفع من المحفظة).</p></div>';
        }
    }

    /**
     * إنشاء منتج الشحن لو مش موجود
     */
    protected function maybe_create_topup_product() {
        $product_id = get_option( self::OPTION_PRODUCT_ID );
        if ( $product_id && get_post( $product_id ) ) {
            return;
        }

        $product = array(
            'post_title'   => 'شحن رصيد محفظة SityCloud',
            'post_content' => 'منتج افتراضي لشحن رصيد محفظة العميل.',
            'post_status'  => 'publish',
            'post_type'    => 'product',
        );

        $product_id = wp_insert_post( $product );

        if ( $product_id && ! is_wp_error( $product_id ) ) {
            update_post_meta( $product_id, '_virtual', 'yes' );
            update_post_meta( $product_id, '_downloadable', 'no' );
            update_post_meta( $product_id, '_sold_individually', 'yes' );
            update_post_meta( $product_id, '_manage_stock', 'no' );
            update_post_meta( $product_id, '_stock_status', 'instock' );
            update_post_meta( $product_id, '_price', '0' ); // السعر يتحدّد ديناميكياً حسب مبلغ الشحن
            update_option( self::OPTION_PRODUCT_ID, $product_id );
        }
    }

    /**
     * شورت كود [sitycloud_wallet]
     */
    public function wallet_shortcode( $atts ) {
        if ( ! is_user_logged_in() ) {
            return '<div class="sity-wallet-box sity-wallet-message">من فضلك سجّل دخولك لمشاهدة رصيدك ومعاملاتك.</div>';
        }

        if ( ! class_exists( 'WooCommerce' ) ) {
            return '<div class="sity-wallet-box sity-wallet-message">يجب تفعيل WooCommerce لاستخدام نظام المحفظة.</div>';
        }

        $user_id      = get_current_user_id();
        $balance      = $this->get_user_balance( $user_id );
        $transactions = $this->get_user_transactions( $user_id );
        $total_paid   = $this->get_user_total_paid( $user_id );

        ob_start();
        ?>
        <style>
            .sity-wallet-wrapper {
                max-width: 1000px;
                margin: 40px auto;
                font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
            }
            .sity-wallet-card {
                background: linear-gradient(135deg, rgba(255,255,255,0.9), rgba(245,238,255,0.95));
                border-radius: 24px;
                padding: 24px 28px;
                box-shadow: 0 18px 40px rgba(80, 45, 140, 0.18);
                border: 1px solid rgba(160, 120, 255, 0.18);
                backdrop-filter: blur(14px);
                margin-bottom: 24px;
            }
            .sity-wallet-header {
                display: flex;
                justify-content: space-between;
                align-items: center;
                gap: 18px;
                margin-bottom: 18px;
            }
            .sity-wallet-title {
                font-size: 1.3rem;
                font-weight: 700;
                color: #3d266e;
            }
            .sity-wallet-sub {
                font-size: 0.9rem;
                color: #6e5e9c;
            }
            .sity-wallet-badge {
                padding: 6px 12px;
                border-radius: 999px;
                background: rgba(134, 86, 255, 0.08);
                color: #6b3df5;
                font-size: 0.8rem;
                font-weight: 600;
            }
            .sity-wallet-stats {
                display: grid;
                grid-template-columns: repeat(3, minmax(0,1fr));
                gap: 14px;
                margin: 12px 0 6px;
            }
            .sity-wallet-stat {
                padding: 12px 14px;
                border-radius: 18px;
                background: rgba(255,255,255,0.9);
                border: 1px solid rgba(180, 150, 255, 0.25);
            }
            .sity-wallet-stat-label {
                font-size: 0.8rem;
                color: #7b6aa8;
                margin-bottom: 4px;
            }
            .sity-wallet-stat-value {
                font-size: 1.1rem;
                font-weight: 700;
                color: #3b236e;
            }
            .sity-wallet-stat-note {
                font-size: 0.75rem;
                color: #9a8eca;
            }
            .sity-wallet-topup {
                margin-top: 12px;
                padding: 16px;
                border-radius: 18px;
                background: rgba(248,245,255,0.95);
                border: 1px dashed rgba(165, 125, 255, 0.6);
                display: flex;
                flex-wrap: wrap;
                gap: 12px;
                align-items: center;
            }
            .sity-wallet-topup label {
                font-size: 0.85rem;
                color: #49337e;
                font-weight: 600;
            }
            .sity-wallet-topup input[type="number"] {
                padding: 9px 12px;
                border-radius: 999px;
                border: 1px solid rgba(170,140,255,0.6);
                background: rgba(255,255,255,0.9);
                min-width: 120px;
                font-size: 0.9rem;
            }
            .sity-wallet-topup input[type="number"]:focus {
                outline: none;
                border-color: #7b4cff;
                box-shadow: 0 0 0 2px rgba(123,76,255,0.2);
            }
            .sity-wallet-btn {
                padding: 9px 18px;
                border-radius: 999px;
                border: none;
                cursor: pointer;
                font-size: 0.9rem;
                font-weight: 600;
                display: inline-flex;
                align-items: center;
                gap: 8px;
                background: linear-gradient(135deg,#7b4cff,#b37dff);
                color: #ffffff;
                box-shadow: 0 10px 25px rgba(110,70,230,0.35);
                transition: transform 0.1s ease, box-shadow 0.1s ease;
            }
            .sity-wallet-btn:hover {
                transform: translateY(-1px);
                box-shadow: 0 14px 28px rgba(110,70,230,0.4);
            }
            .sity-wallet-btn:active {
                transform: translateY(0);
                box-shadow: 0 6px 16px rgba(110,70,230,0.3);
            }
            .sity-wallet-table-wrap {
                overflow-x: auto;
                border-radius: 18px;
                border: 1px solid rgba(205,190,255,0.7);
                background: rgba(255,255,255,0.92);
            }
            table.sity-wallet-table {
                width: 100%;
                border-collapse: collapse;
                min-width: 600px;
            }
            table.sity-wallet-table thead {
                background: linear-gradient(135deg, rgba(122,80,255,0.09), rgba(189,153,255,0.15));
            }
            table.sity-wallet-table th,
            table.sity-wallet-table td {
                padding: 10px 12px;
                text-align: right;
                font-size: 0.85rem;
            }
            table.sity-wallet-table th {
                font-weight: 700;
                color: #3b236e;
            }
            table.sity-wallet-table tbody tr:nth-child(even) {
                background: rgba(248,245,255,0.9);
            }
            table.sity-wallet-table tbody tr:nth-child(odd) {
                background: rgba(255,255,255,0.98);
            }
            .sity-wallet-chip-credit {
                display: inline-block;
                padding: 3px 9px;
                border-radius: 999px;
                background: rgba(60,180,120,0.1);
                color: #168a55;
                font-size: 0.78rem;
                font-weight: 600;
            }
            .sity-wallet-chip-debit {
                display: inline-block;
                padding: 3px 9px;
                border-radius: 999px;
                background: rgba(235,70,90,0.1);
                color: #c92a4a;
                font-size: 0.78rem;
                font-weight: 600;
            }
            .sity-wallet-amount-credit {
                color: #168a55;
                font-weight: 600;
            }
            .sity-wallet-amount-debit {
                color: #c92a4a;
                font-weight: 600;
            }
            .sity-wallet-empty {
                padding: 12px 16px;
                font-size: 0.9rem;
                color: #7a6db0;
            }
            .sity-wallet-message {
                max-width: 700px;
                margin: 20px auto;
                padding: 16px 20px;
                border-radius: 16px;
                background: rgba(255,255,255,0.98);
                border: 1px solid rgba(200,180,255,0.7);
                color: #4a347f;
            }
            @media (max-width: 768px) {
                .sity-wallet-stats {
                    grid-template-columns: repeat(1,minmax(0,1fr));
                }
                .sity-wallet-header {
                    flex-direction: column;
                    align-items: flex-start;
                }
            }
        </style>

        <div class="sity-wallet-wrapper">
            <div class="sity-wallet-card">
                <div class="sity-wallet-header">
                    <div>
                        <div class="sity-wallet-title">محفظتك في Sity Cloud</div>
                        <div class="sity-wallet-sub">تابع رصيدك، عمليات الشحن، والمدفوعات من مكان واحد.</div>
                    </div>
                    <div class="sity-wallet-badge">عميل: <?php echo esc_html( wp_get_current_user()->display_name ); ?></div>
                </div>

                <div class="sity-wallet-stats">
                    <div class="sity-wallet-stat">
                        <div class="sity-wallet-stat-label">الرصيد الحالي</div>
                        <div class="sity-wallet-stat-value"><?php echo number_format( $balance, 2 ); ?> ج.م</div>
                        <div class="sity-wallet-stat-note">يمكنك استخدامه كوسيلة دفع.</div>
                    </div>
                    <div class="sity-wallet-stat">
                        <div class="sity-wallet-stat-label">إجمالي المدفوعات من المحفظة</div>
                        <div class="sity-wallet-stat-value"><?php echo number_format( $total_paid, 2 ); ?> ج.م</div>
                        <div class="sity-wallet-stat-note">صافي مجموع ما تم سداده من خلال محفظتك.</div>
                    </div>
                    <div class="sity-wallet-stat">
                        <div class="sity-wallet-stat-label">عدد الحركات</div>
                        <div class="sity-wallet-stat-value"><?php echo count( $transactions ); ?></div>
                        <div class="sity-wallet-stat-note">شحن + مدفوعات + تعديل رصيد.</div>
                    </div>
                </div>

                <form class="sity-wallet-topup" method="post">
                    <?php wp_nonce_field( 'sitycloud_wallet_topup', 'sitycloud_wallet_topup_nonce' ); ?>
                    <input type="hidden" name="sitycloud_wallet_topup" value="1">
                    <div>
                        <label for="sitycloud_wallet_topup_amount">عايز تشحن كام في رصيدك؟</label>
                    </div>
                    <div>
                        <input type="number" id="sitycloud_wallet_topup_amount" name="amount" min="1" step="0.01" required>
                    </div>
                    <button type="submit" class="sity-wallet-btn">
                        اشحن رصيدك الآن
                        <span>⚡</span>
                    </button>
                </form>
            </div>

            <div class="sity-wallet-card">
                <div class="sity-wallet-header" style="margin-bottom:14px;">
                    <div>
                        <div class="sity-wallet-title">سجل عمليات المحفظة</div>
                        <div class="sity-wallet-sub">كل عمليات الشحن والمدفوعات والتعديلات مسجّلة هنا.</div>
                    </div>
                </div>

                <div class="sity-wallet-table-wrap">
                    <?php if ( ! empty( $transactions ) ) : ?>
                        <table class="sity-wallet-table">
                            <thead>
                                <tr>
                                    <th>التاريخ</th>
                                    <th>النوع</th>
                                    <th>الوصف</th>
                                    <th>رقم الطلب</th>
                                    <th>المبلغ</th>
                                    <th>الرصيد بعد العملية</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ( $transactions as $tx ) : ?>
                                    <tr>
                                        <td><?php echo esc_html( date_i18n( 'Y-m-d H:i', strtotime( $tx->created_at ) ) ); ?></td>
                                        <td>
                                            <?php
                                            if ( $tx->type === 'credit' ) {
                                                echo '<span class="sity-wallet-chip-credit">شحن / إضافة</span>';
                                            } else {
                                                echo '<span class="sity-wallet-chip-debit">مدفوعة / خصم</span>';
                                            }
                                            ?>
                                        </td>
                                        <td><?php echo esc_html( $tx->description ); ?></td>
                                        <td><?php echo $tx->order_id ? esc_html( '#' . $tx->order_id ) : '-'; ?></td>
                                        <td class="<?php echo $tx->type === 'credit' ? 'sity-wallet-amount-credit' : 'sity-wallet-amount-debit'; ?>">
                                            <?php
                                            $sign = $tx->type === 'credit' ? '+' : '-';
                                            echo $sign . number_format( $tx->amount, 2 ) . ' ج.م';
                                            ?>
                                        </td>
                                        <td><?php echo number_format( $tx->balance_after, 2 ); ?> ج.م</td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else : ?>
                        <div class="sity-wallet-empty">لا توجد أي معاملات حتى الآن. ابدأ بشحن رصيدك أو الدفع باستخدام المحفظة.</div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php

        return ob_get_clean();
    }

    /**
     * معالجة فورم الشحن من الفرونت
     */
    public function handle_frontend_topup() {
        if (
            ! isset( $_POST['sitycloud_wallet_topup'] ) ||
            ! isset( $_POST['sitycloud_wallet_topup_nonce'] )
        ) {
            return;
        }

        if ( ! is_user_logged_in() ) {
            return;
        }

        if ( ! wp_verify_nonce( $_POST['sitycloud_wallet_topup_nonce'], 'sitycloud_wallet_topup' ) ) {
            return;
        }

        if ( ! class_exists( 'WooCommerce' ) || ! function_exists( 'WC' ) ) {
            if ( function_exists( 'wc_add_notice' ) ) {
                wc_add_notice( 'لا يمكن اتمام عملية الشحن الان', 'error' );
            }
            return;
        }

        $amount = isset( $_POST['amount'] ) ? floatval( $_POST['amount'] ) : 0;
        if ( $amount <= 0 ) {
            if ( function_exists( 'wc_add_notice' ) ) {
                wc_add_notice( 'برجاء إدخال مبلغ شحن صحيح.', 'error' );
            }
            return;
        }

        $product_id = get_option( self::OPTION_PRODUCT_ID );
        if ( ! $product_id || ! get_post( $product_id ) ) {
            $this->maybe_create_topup_product();
            $product_id = get_option( self::OPTION_PRODUCT_ID );
        }

        if ( ! $product_id ) {
            if ( function_exists( 'wc_add_notice' ) ) {
                wc_add_notice( 'حدث خطأ في إعداد منتج شحن الرصيد، تواصل مع الدعم.', 'error' );
            }
            return;
        }

        // إعداد عربة الشراء
        WC()->cart->empty_cart();

        $cart_item_data = array(
            'sitycloud_topup_amount' => $amount,
            'sitycloud_topup_user'   => get_current_user_id(),
        );

        WC()->cart->add_to_cart( $product_id, 1, 0, array(), $cart_item_data );

        if ( function_exists( 'wc_add_notice' ) ) {
            wc_add_notice( 'تم إضافة عملية شحن الرصيد إلى سلة المشتريات، أكمل عملية الدفع.', 'success' );
        }

        wp_safe_redirect( wc_get_checkout_url() );
        exit;
    }

    /**
     * ضبط سعر المنتج في الكارت بناءً على مبلغ الشحن
     */
    public function set_cart_item_price_from_meta( $cart ) {
        if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
            return;
        }

        if ( empty( $cart ) || ! is_object( $cart ) ) {
            return;
        }

        foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
            if ( isset( $cart_item['sitycloud_topup_amount'] ) ) {
                $amount = floatval( $cart_item['sitycloud_topup_amount'] );
                if ( $amount > 0 && isset( $cart_item['data'] ) && is_object( $cart_item['data'] ) ) {
                    $cart_item['data']->set_price( $amount );
                }
            }
        }
    }

    /**
     * تخزين بيانات الشحن في الـ order item
     */
    public function save_cart_meta_to_order_item( $item, $cart_item_key, $values, $order ) {
        if ( isset( $values['sitycloud_topup_amount'] ) ) {
            $item->add_meta_data( 'sitycloud_topup_amount', $values['sitycloud_topup_amount'], true );
        }
        if ( isset( $values['sitycloud_topup_user'] ) ) {
            $item->add_meta_data( 'sitycloud_topup_user', $values['sitycloud_topup_user'], true );
        }
    }

    /**
     * عند اكتمال الطلب → زيادة رصيد المحفظة
     */
    public function handle_order_completed_topup( $order_id ) {
        if ( ! $order_id ) {
            return;
        }

        $order = wc_get_order( $order_id );
        if ( ! $order ) {
            return;
        }

        foreach ( $order->get_items() as $item_id => $item ) {
            $topup_amount = $item->get_meta( 'sitycloud_topup_amount', true );
            $topup_user   = $item->get_meta( 'sitycloud_topup_user', true );

            if ( $topup_amount && $topup_user ) {
                $amount = floatval( $topup_amount );
                if ( $amount <= 0 ) {
                    continue;
                }

                $this->change_user_balance(
                    $topup_user,
                    $amount,
                    'credit',
                    sprintf( 'شحن رصيد عبر الطلب #%d', $order_id ),
                    $order_id
                );
            }
        }
    }

    /**
     * تسجيل بوابة الدفع الخاصة بالمحفظة
     */
    public function register_wallet_gateway( $gateways ) {
        if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
            return $gateways;
        }

        // الكلاس متعرف تحت
        $gateways[] = 'WC_Gateway_Sitycloud_Wallet';
        return $gateways;
    }

    /**
     * جلب رصيد المستخدم
     */
    public function get_user_balance( $user_id ) {
        $balance = get_user_meta( $user_id, self::META_BALANCE_KEY, true );
        if ( $balance === '' || $balance === null ) {
            $balance = 0;
        }
        return floatval( $balance );
    }

    /**
     * تغيير رصيد المستخدم + تسجيل الحركة
     */
    public function change_user_balance( $user_id, $amount, $type = 'credit', $description = '', $order_id = null ) {
        global $wpdb;

        $current_balance = $this->get_user_balance( $user_id );
        $amount          = floatval( $amount );

        if ( $type === 'debit' ) {
            $new_balance = $current_balance - $amount;
        } else {
            $new_balance = $current_balance + $amount;
        }

        if ( $new_balance < 0 ) {
            $new_balance = 0;
        }

        update_user_meta( $user_id, self::META_BALANCE_KEY, $new_balance );

        $table_name = $wpdb->prefix . self::TABLE_NAME;

        $wpdb->insert(
            $table_name,
            array(
                'user_id'       => $user_id,
                'type'          => $type,
                'amount'        => $amount,
                'description'   => $description,
                'order_id'      => $order_id ? intval( $order_id ) : null,
                'balance_after' => $new_balance,
                'created_at'    => current_time( 'mysql' ),
            ),
            array( '%d', '%s', '%f', '%s', '%d', '%f', '%s' )
        );

        return $new_balance;
    }

    /**
     * كل معاملات مستخدم معيّن
     */
    public function get_user_transactions( $user_id ) {
        global $wpdb;

        $table_name = $wpdb->prefix . self::TABLE_NAME;

        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$table_name} WHERE user_id = %d ORDER BY created_at DESC",
                $user_id
            )
        );

        return $results ? $results : array();
    }

    /**
     * إجمالي المبالغ المدفوعة (خصم) من المحفظة
     */
    public function get_user_total_paid( $user_id ) {
        global $wpdb;

        $table_name = $wpdb->prefix . self::TABLE_NAME;

        $total = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT SUM(amount) FROM {$table_name} WHERE user_id = %d AND type = 'debit'",
                $user_id
            )
        );

        return $total ? floatval( $total ) : 0;
    }

    /**
     * قائمة الأدمن
     */
    public function register_admin_menu() {
        add_menu_page(
            'محفظة SityCloud',
            'محفظة SityCloud',
            'manage_woocommerce',
            'sitycloud-wallet',
            array( $this, 'render_admin_page' ),
            'dashicons-money-alt',
            56
        );
    }

    /**
     * صفحة الأدمن لرصيد العملاء + المعاملات
     */
    public function render_admin_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) && ! current_user_can( 'manage_options' ) ) {
            wp_die( 'غير مسموح لك بالدخول إلى هذه الصفحة.' );
        }

        $selected_user_id = isset( $_GET['user_id'] ) ? intval( $_GET['user_id'] ) : 0;

        ?>
        <div class="wrap">
            <h1 style="margin-bottom:16px;">محفظة SityCloud - رصيد العملاء والمعاملات</h1>

            <style>
                .sity-admin-card {
                    background: #ffffff;
                    border-radius: 14px;
                    padding: 16px 18px;
                    border: 1px solid #e3ddff;
                    box-shadow: 0 4px 14px rgba(60,35,120,0.06);
                    margin-bottom: 18px;
                }
                .sity-admin-table {
                    width: 100%;
                    border-collapse: collapse;
                }
                .sity-admin-table th,
                .sity-admin-table td {
                    padding: 8px 10px;
                    border-bottom: 1px solid #f0ecff;
                    text-align: right;
                }
                .sity-admin-table th {
                    background: #f7f4ff;
                    font-weight: 600;
                }
                .sity-admin-badge {
                    display: inline-block;
                    padding: 3px 8px;
                    border-radius: 999px;
                    background: rgba(123,76,255,0.1);
                    color: #5c34d6;
                    font-size: 11px;
                    font-weight: 600;
                }
                .sity-admin-btn {
                    display: inline-block;
                    padding: 6px 10px;
                    border-radius: 999px;
                    background: #7b4cff;
                    color: #ffffff;
                    text-decoration: none;
                    font-size: 11px;
                    font-weight: 600;
                }
                .sity-admin-btn:hover {
                    background: #6332da;
                }
                .sity-admin-input {
                    padding: 4px 6px;
                    border-radius: 6px;
                    border: 1px solid #d7cdfa;
                }
                .sity-admin-select {
                    padding: 4px 6px;
                    border-radius: 6px;
                    border: 1px solid #d7cdfa;
                }
                .sity-admin-submit {
                    padding: 5px 10px;
                    border-radius: 999px;
                    border: none;
                    background: #3dba7a;
                    color: #ffffff;
                    font-size: 11px;
                    font-weight: 600;
                    cursor: pointer;
                }
            </style>

            <div class="sity-admin-card">
                <h2 style="margin-top:0;">جميع العملاء وأرصدة المحفظة</h2>
                <?php
                $args  = array(
                    'meta_key'     => self::META_BALANCE_KEY,
                    'meta_compare' => 'EXISTS',
                    'number'       => 200,
                );
                $users = get_users( $args );
                ?>
                <table class="sity-admin-table">
                    <thead>
                        <tr>
                            <th>المستخدم</th>
                            <th>الإيميل</th>
                            <th>الرصيد الحالي</th>
                            <th>عرض المعاملات</th>
                            <th>تعديل الرصيد</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ( ! empty( $users ) ) : ?>
                            <?php foreach ( $users as $user ) : ?>
                                <?php $balance = $this->get_user_balance( $user->ID ); ?>
                                <tr>
                                    <td>
                                        <?php echo esc_html( $user->display_name ); ?><br>
                                        <span class="sity-admin-badge">ID: <?php echo esc_html( $user->ID ); ?></span>
                                    </td>
                                    <td><?php echo esc_html( $user->user_email ); ?></td>
                                    <td><strong><?php echo number_format( $balance, 2 ); ?> ج.م</strong></td>
                                    <td>
                                        <a class="sity-admin-btn" href="<?php echo esc_url( admin_url( 'admin.php?page=sitycloud-wallet&user_id=' . $user->ID ) ); ?>">
                                            عرض معاملات المستخدم
                                        </a>
                                    </td>
                                    <td>
                                        <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                                            <?php wp_nonce_field( 'sitycloud_wallet_adjust', 'sitycloud_wallet_adjust_nonce' ); ?>
                                            <input type="hidden" name="action" value="sitycloud_wallet_adjust">
                                            <input type="hidden" name="user_id" value="<?php echo esc_attr( $user->ID ); ?>">
                                            <select name="op" class="sity-admin-select">
                                                <option value="credit">إضافة</option>
                                                <option value="debit">خصم</option>
                                            </select>
                                            <input type="number" step="0.01" name="amount" class="sity-admin-input" style="width:80px;" required>
                                            <input type="text" name="note" class="sity-admin-input" placeholder="وصف" style="width:140px;">
                                            <button type="submit" class="sity-admin-submit">تنفيذ</button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else : ?>
                            <tr><td colspan="5">لا يوجد أي عملاء لديهم رصيد حتى الآن.</td></tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <?php if ( $selected_user_id ) : ?>
                <?php
                $user = get_user_by( 'id', $selected_user_id );
                if ( $user ) :
                    $transactions = $this->get_user_transactions( $selected_user_id );
                    ?>
                    <div class="sity-admin-card">
                        <h2>معاملات المستخدم: <?php echo esc_html( $user->display_name ); ?> (ID: <?php echo esc_html( $user->ID ); ?>)</h2>
                        <p>الرصيد الحالي: <strong><?php echo number_format( $this->get_user_balance( $selected_user_id ), 2 ); ?> ج.م</strong></p>

                        <?php if ( ! empty( $transactions ) ) : ?>
                            <table class="sity-admin-table">
                                <thead>
                                    <tr>
                                        <th>التاريخ</th>
                                        <th>النوع</th>
                                        <th>الوصف</th>
                                        <th>رقم الطلب</th>
                                        <th>المبلغ</th>
                                        <th>الرصيد بعد العملية</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ( $transactions as $tx ) : ?>
                                        <tr>
                                            <td><?php echo esc_html( date_i18n( 'Y-m-d H:i', strtotime( $tx->created_at ) ) ); ?></td>
                                            <td><?php echo $tx->type === 'credit' ? 'شحن / إضافة' : 'مدفوعة / خصم'; ?></td>
                                            <td><?php echo esc_html( $tx->description ); ?></td>
                                            <td><?php echo $tx->order_id ? esc_html( '#' . $tx->order_id ) : '-'; ?></td>
                                            <td><?php echo number_format( $tx->amount, 2 ); ?> ج.م</td>
                                            <td><?php echo number_format( $tx->balance_after, 2 ); ?> ج.م</td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php else : ?>
                            <p>لا توجد أي معاملات لهذا المستخدم حتى الآن.</p>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * معالجة تعديل الرصيد من لوحة التحكم
     */
    public function handle_admin_adjust_balance() {
        if ( ! current_user_can( 'manage_woocommerce' ) && ! current_user_can( 'manage_options' ) ) {
            wp_die( 'غير مصرح.' );
        }

        if ( ! isset( $_POST['sitycloud_wallet_adjust_nonce'] ) ||
             ! wp_verify_nonce( $_POST['sitycloud_wallet_adjust_nonce'], 'sitycloud_wallet_adjust' )
        ) {
            wp_die( 'طلب غير صالح.' );
        }

        $user_id = isset( $_POST['user_id'] ) ? intval( $_POST['user_id'] ) : 0;
        $op      = isset( $_POST['op'] ) ? sanitize_text_field( $_POST['op'] ) : 'credit';
        $amount  = isset( $_POST['amount'] ) ? floatval( $_POST['amount'] ) : 0;
        $note    = isset( $_POST['note'] ) ? sanitize_text_field( $_POST['note'] ) : '';

        if ( $user_id <= 0 || $amount <= 0 ) {
            wp_redirect( admin_url( 'admin.php?page=sitycloud-wallet' ) );
            exit;
        }

        $type = $op === 'debit' ? 'debit' : 'credit';

        $this->change_user_balance(
            $user_id,
            $amount,
            $type,
            $note ? $note : ( $type === 'credit' ? 'تعديل رصيد (إضافة يدويًا)' : 'تعديل رصيد (خصم يدويًا)' ),
            null
        );

        wp_redirect( admin_url( 'admin.php?page=sitycloud-wallet&user_id=' . $user_id ) );
        exit;
    }
}

/**
 * بوابة الدفع بمحفظة SityCloud
 */
if ( class_exists( 'WC_Payment_Gateway' ) ) {

    class WC_Gateway_Sitycloud_Wallet extends WC_Payment_Gateway {

        public function __construct() {
            $this->id                 = 'sitycloud_wallet';
            $this->method_title       = 'محفظة SityCloud';
            $this->method_description = 'الدفع باستخدام رصيد المحفظة في SityCloud.';
            $this->has_fields         = false;
            $this->icon               = '';

            $this->init_form_fields();
            $this->init_settings();

            $this->title       = $this->get_option( 'title' );
            $this->description = $this->get_option( 'description' );
            $this->enabled     = $this->get_option( 'enabled' );

            add_action(
                'woocommerce_update_options_payment_gateways_' . $this->id,
                array( $this, 'process_admin_options' )
            );
        }

        public function init_form_fields() {
            $this->form_fields = array(
                'enabled'    => array(
                    'title'   => 'تفعيل',
                    'type'    => 'checkbox',
                    'label'   => 'تفعيل بوابة الدفع بمحفظة SityCloud',
                    'default' => 'yes',
                ),
                'title'      => array(
                    'title'   => 'عنوان طريقة الدفع',
                    'type'    => 'text',
                    'default' => 'الدفع من محفظة SityCloud',
                ),
                'description'=> array(
                    'title'   => 'الوصف',
                    'type'    => 'textarea',
                    'default' => 'استخدم رصيد محفظتك في SityCloud لدفع هذا الطلب.',
                ),
            );
        }

        public function is_available() {
            if ( 'yes' !== $this->enabled ) {
                return false;
            }
            if ( ! is_user_logged_in() ) {
                return false;
            }
            $user_id = get_current_user_id();
            $plugin  = SityCloud_Wallet_Plugin::get_instance();
            $balance = $plugin->get_user_balance( $user_id );
            if ( $balance <= 0 ) {
                return false;
            }
            return parent::is_available();
        }

        public function process_payment( $order_id ) {
            $order = wc_get_order( $order_id );
            if ( ! $order ) {
                wc_add_notice( 'حدث خطأ أثناء معالجة الطلب.', 'error' );
                return;
            }

            if ( ! is_user_logged_in() ) {
                wc_add_notice( 'يجب تسجيل الدخول لاستخدام المحفظة.', 'error' );
                return;
            }

            $user_id = get_current_user_id();
            $plugin  = SityCloud_Wallet_Plugin::get_instance();
            $balance = $plugin->get_user_balance( $user_id );
            $total   = floatval( $order->get_total() );

            if ( $balance < $total ) {
                wc_add_notice( 'رصيدك في المحفظة غير كافٍ لدفع هذا الطلب.', 'error' );
                return;
            }

            // خصم الرصيد وتسجيل العملية
            $plugin->change_user_balance(
                $user_id,
                $total,
                'debit',
                sprintf( 'دفع طلب #%d من المحفظة', $order_id ),
                $order_id
            );

            $order->payment_complete();
            $order->add_order_note( 'تم دفع الطلب من خلال محفظة SityCloud.' );

            WC()->cart->empty_cart();

            return array(
                'result'   => 'success',
                'redirect' => $order->get_checkout_order_received_url(),
            );
        }
    }
}

/**
 * تفعيل البلجن
 */
register_activation_hook( __FILE__, array( 'SityCloud_Wallet_Plugin', 'activate' ) );

/**
 * تشغيل البلجن
 */
SityCloud_Wallet_Plugin::init();

